"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CMS = exports.updateContent = exports.sortLogs = exports.applyLog = exports.resolveIndexFromTraversalResult = exports.resolveTraversalAddressFromPath = exports.resolveAddressForTraversal = void 0;
const traverse_1 = __importDefault(require("traverse"));
const idResolutionExpressions = {
    default: () => /^\[.*\]/gm,
};
/**
 * Resolves a complete object address to the current index
 * @param {traverse.Traverse<any>} traverseResult
 * @param {string} inputAddress
 * @return {string}
 */
function resolveAddressForTraversal(traverseResult, inputAddress) {
    const pathParts = inputAddress.split('.');
    let i = 0;
    for (i; i < pathParts.length; i++) {
        const currentPath = pathParts[i];
        if (idResolutionExpressions.default().test(currentPath) === true) {
            const parentAddress = pathParts.slice(0, i);
            const resolvedId = resolveIndexFromTraversalResult(traverseResult, resolveTraversalAddressFromPath(parentAddress), currentPath);
            pathParts[i] = String(resolvedId);
        }
    }
    return resolveTraversalAddressFromPath(pathParts);
}
exports.resolveAddressForTraversal = resolveAddressForTraversal;
/**
 * Resolves traversal paths to address
 * @param {string[]} paths
 * @return {string}
 */
function resolveTraversalAddressFromPath(paths) {
    return paths.join('.');
}
exports.resolveTraversalAddressFromPath = resolveTraversalAddressFromPath;
/**
 * Resolves index of array item from the ID template
 * @param {traverse.Traverse<any>} traverseResult
 * @param {string} path
 * @param {string} query
 * @return {number}
 */
function resolveIndexFromTraversalResult(traverseResult, path, query) {
    let id = null;
    if (idResolutionExpressions.default().test(query) === false) {
        return Number(query);
    }
    try {
        const results = idResolutionExpressions.default().exec(query);
        if (results && results.length > 0) {
            id = results[0].replace('[', '').replace(']', '');
        }
    }
    catch (e) {
        console.error(e);
    }
    const target = traverseResult.get(path.split('.').filter(Boolean));
    if (Array.isArray(target) && target.length > 0) {
        const index = target.findIndex((t) => {
            try {
                if (t.id) {
                    if (String(t.id) === String(id)) {
                        return true;
                    }
                }
            }
            catch (e) {
                console.error(e);
            }
            return false;
        });
        return index;
    }
    return -1;
}
exports.resolveIndexFromTraversalResult = resolveIndexFromTraversalResult;
function applyLog(content, key, val, mode, track) {
    const traverseResult = (0, traverse_1.default)(content);
    const resolvedKey = resolveAddressForTraversal(traverseResult, key);
    const paths = traverseResult.paths().filter((p) => p.length > 0);
    let i = 0;
    let hasValueSet = false;
    for (i = 0; i < paths.length; i++) {
        const address = resolveTraversalAddressFromPath(paths[i]);
        if (address === resolvedKey) {
            if (track) {
                const currentVal = traverseResult.get(paths[i]);
                track(currentVal, typeof currentVal);
            }
            traverseResult.set(paths[i], val);
            hasValueSet = true;
            break;
        }
    }
    if (hasValueSet === false) {
        if (Boolean(resolvedKey)) {
            /** Creating a new key */
            traverseResult.set(String(resolvedKey).split('.'), val);
        }
    }
    return content;
}
exports.applyLog = applyLog;
function* infinite() {
    let index = 0;
    while (true) {
        yield index++;
    }
    return index;
}
const generator = infinite();
function sortLogs(actions) {
    actions = actions.sort((a, b) => {
        if (!isNaN(a.id) && !isNaN(b.id)) {
            if (a.id > b.id) {
                return 1;
            }
            return -1;
        }
        return undefined;
    });
    return actions;
}
exports.sortLogs = sortLogs;
function updateContent(content, actions, mode) {
    actions = sortLogs(actions);
    let i = 0;
    for (i = 0; i < actions.length; i++) {
        actions[i].id = generator.next().value;
        content = applyLog(content, actions[i].key, actions[i].val, mode, (prevVal, prevValType) => {
            actions[i].prevValueType = prevValType;
            if (actions[i].prevValueType === 'string') {
                // Prev value enabled only for string now
                actions[i].previousValue = prevVal;
            }
        });
    }
    return {
        appliedActions: actions,
        content,
    };
}
exports.updateContent = updateContent;
exports.CMS = {
    applyLog,
    updateContent,
};
