"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClockTable = exports.ClockTableEntry = void 0;
const clock_1 = require("../../clock");
class ClockTableEntry {
    constructor(index, id) {
        this.index = index;
        this.id = id;
    }
}
exports.ClockTableEntry = ClockTableEntry;
class ClockTable {
    constructor() {
        this.bySid = new Map();
        this.byIdx = [];
    }
    static from(clock) {
        const table = new ClockTable();
        table.push(new clock_1.Timestamp(clock.sid, clock.time - 1));
        clock.peers.forEach((id) => table.push(id));
        return table;
    }
    static decode(reader) {
        const clockTable = new ClockTable();
        const length = reader.vu57();
        const tuple = reader.u53vu39();
        clockTable.push(new clock_1.Timestamp(tuple[0], tuple[1]));
        for (let i = 1; i < length; i++) {
            const tuple = reader.u53vu39();
            clockTable.push(new clock_1.Timestamp(tuple[0], tuple[1]));
        }
        return clockTable;
    }
    parseField(field) {
        const underscoreIndex = field.indexOf('_');
        const relativeSid = parseInt(field.slice(0, underscoreIndex), 36);
        const time = parseInt(field.slice(underscoreIndex + 1), 36);
        const clock = this.byIdx[relativeSid];
        return new clock_1.Timestamp(clock.sid, time);
    }
    push(id) {
        const byIdx = this.byIdx;
        const index = byIdx.length;
        byIdx.push(id);
        this.bySid.set(id.sid, new ClockTableEntry(index, id));
    }
    getBySid(sid) {
        const entry = this.bySid.get(sid);
        if (!entry)
            throw new Error('CLOCK_NOT_FOUND');
        return entry;
    }
    write(writer) {
        const table = this.byIdx;
        const length = table.length;
        writer.vu39(length);
        for (let i = 0; i < length; i++) {
            const clock = table[i];
            writer.u53vu39(clock.sid, clock.time);
        }
    }
}
exports.ClockTable = ClockTable;
