"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Model = exports.UNDEFINED = void 0;
const tslib_1 = require("tslib");
const operations = tslib_1.__importStar(require("../../json-crdt-patch/operations"));
const ConNode_1 = require("../nodes/con/ConNode");
const shared_1 = require("../codec/structural/binary/shared");
const clock_1 = require("../../json-crdt-patch/clock");
const ModelApi_1 = require("./api/ModelApi");
const constants_1 = require("../../json-crdt-patch/constants");
const util_1 = require("./util");
const nodes_1 = require("../nodes");
const printTree_1 = require("../../util/print/printTree");
const Extensions_1 = require("../extensions/Extensions");
const AvlMap_1 = require("../../util/trees/avl/AvlMap");
exports.UNDEFINED = new ConNode_1.ConNode(constants_1.ORIGIN, undefined);
class Model {
    static withLogicalClock(clockOrSessionId) {
        const clock = typeof clockOrSessionId === 'number'
            ? new clock_1.VectorClock(clockOrSessionId, 1)
            : clockOrSessionId || new clock_1.VectorClock((0, util_1.randomSessionId)(), 1);
        return new Model(clock);
    }
    static withServerClock(time = 0) {
        const clock = new clock_1.ServerVectorClock(1, time);
        return new Model(clock);
    }
    static fromBinary(data) {
        return shared_1.decoder.decode(data);
    }
    constructor(clock) {
        this.root = new nodes_1.RootNode(this, constants_1.ORIGIN);
        this.index = new AvlMap_1.AvlMap(clock_1.compare);
        this.ext = new Extensions_1.Extensions();
        this.tick = 0;
        this.onchange = undefined;
        this.clock = clock;
        if (!clock.time)
            clock.time = 1;
    }
    get api() {
        if (!this._api)
            this._api = new ModelApi_1.ModelApi(this);
        return this._api;
    }
    get find() {
        return this.api.r.proxy();
    }
    applyBatch(patches) {
        const length = patches.length;
        for (let i = 0; i < length; i++)
            this.applyPatch(patches[i]);
    }
    applyPatch(patch) {
        const ops = patch.ops;
        const { length } = ops;
        for (let i = 0; i < length; i++)
            this.applyOperation(ops[i]);
        this.tick++;
        this.onchange?.();
    }
    applyOperation(op) {
        this.clock.observe(op.id, op.span());
        const index = this.index;
        if (op instanceof operations.InsStrOp) {
            const node = index.get(op.obj);
            if (node instanceof nodes_1.StrNode)
                node.ins(op.ref, op.id, op.data);
        }
        else if (op instanceof operations.NewObjOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new nodes_1.ObjNode(this, id));
        }
        else if (op instanceof operations.NewArrOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new nodes_1.ArrNode(this, id));
        }
        else if (op instanceof operations.NewStrOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new nodes_1.StrNode(id));
        }
        else if (op instanceof operations.NewValOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new nodes_1.ValNode(this, id, constants_1.ORIGIN));
        }
        else if (op instanceof operations.NewConOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new ConNode_1.ConNode(id, op.val));
        }
        else if (op instanceof operations.InsObjOp) {
            const node = index.get(op.obj);
            const tuples = op.data;
            const length = tuples.length;
            if (node instanceof nodes_1.ObjNode) {
                for (let i = 0; i < length; i++) {
                    const tuple = tuples[i];
                    const valueNode = index.get(tuple[1]);
                    if (!valueNode)
                        continue;
                    if (node.id.time >= tuple[1].time)
                        continue;
                    const old = node.put(tuple[0] + '', valueNode.id);
                    if (old)
                        this.deleteNodeTree(old);
                }
            }
        }
        else if (op instanceof operations.InsVecOp) {
            const node = index.get(op.obj);
            const tuples = op.data;
            const length = tuples.length;
            if (node instanceof nodes_1.VecNode) {
                for (let i = 0; i < length; i++) {
                    const tuple = tuples[i];
                    const valueNode = index.get(tuple[1]);
                    if (!valueNode)
                        continue;
                    if (node.id.time >= tuple[1].time)
                        continue;
                    const old = node.put(Number(tuple[0]), valueNode.id);
                    if (old)
                        this.deleteNodeTree(old);
                }
            }
        }
        else if (op instanceof operations.InsValOp) {
            const obj = op.obj;
            const node = obj.sid === 0 && obj.time === 0 ? this.root : index.get(obj);
            if (node instanceof nodes_1.ValNode) {
                const newValue = index.get(op.val);
                if (newValue) {
                    const old = node.set(op.val);
                    if (old)
                        this.deleteNodeTree(old);
                }
            }
        }
        else if (op instanceof operations.InsArrOp) {
            const node = index.get(op.obj);
            if (node instanceof nodes_1.ArrNode) {
                const nodes = [];
                const data = op.data;
                const length = data.length;
                for (let i = 0; i < length; i++) {
                    const stamp = data[i];
                    const valueNode = index.get(stamp);
                    if (!valueNode)
                        continue;
                    if (node.id.time >= stamp.time)
                        continue;
                    nodes.push(stamp);
                }
                if (nodes.length)
                    node.ins(op.ref, op.id, nodes);
            }
        }
        else if (op instanceof operations.DelOp) {
            const node = index.get(op.obj);
            if (node instanceof nodes_1.ArrNode) {
                const length = op.what.length;
                for (let i = 0; i < length; i++) {
                    const span = op.what[i];
                    for (let j = 0; j < span.span; j++) {
                        const id = node.getById(new clock_1.Timestamp(span.sid, span.time + j));
                        if (id)
                            this.deleteNodeTree(id);
                    }
                }
                node.delete(op.what);
            }
            else if (node instanceof nodes_1.StrNode)
                node.delete(op.what);
            else if (node instanceof nodes_1.BinNode)
                node.delete(op.what);
        }
        else if (op instanceof operations.NewBinOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new nodes_1.BinNode(id));
        }
        else if (op instanceof operations.InsBinOp) {
            const node = index.get(op.obj);
            if (node instanceof nodes_1.BinNode)
                node.ins(op.ref, op.id, op.data);
        }
        else if (op instanceof operations.NewVecOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new nodes_1.VecNode(this, id));
        }
    }
    deleteNodeTree(value) {
        const isSystemNode = value.sid === 0;
        if (isSystemNode)
            return;
        const node = this.index.get(value);
        if (!node)
            return;
        node.children((child) => this.deleteNodeTree(child.id));
        this.index.del(value);
    }
    fork(sessionId = (0, util_1.randomSessionId)()) {
        const copy = Model.fromBinary(this.toBinary());
        if (copy.clock.sid !== sessionId && copy.clock instanceof clock_1.VectorClock)
            copy.clock = copy.clock.fork(sessionId);
        copy.ext = this.ext;
        return copy;
    }
    clone() {
        return this.fork(this.clock.sid);
    }
    view() {
        return this.root.view();
    }
    toBinary() {
        return shared_1.encoder.encode(this);
    }
    setSchema(schema) {
        if (this.clock.time < 2)
            this.api.root(schema);
        return this;
    }
    toString(tab = '') {
        const nl = () => '';
        const hasExtensions = this.ext.size() > 0;
        return (this.constructor.name +
            (0, printTree_1.printTree)(tab, [
                (tab) => this.root.toString(tab),
                nl,
                (tab) => {
                    const nodes = [];
                    this.index.forEach((item) => nodes.push(item.v));
                    return (`Index (${nodes.length} nodes)` +
                        (nodes.length
                            ? (0, printTree_1.printTree)(tab, nodes.map((node) => (tab) => `${node.constructor.name} ${(0, clock_1.toDisplayString)(node.id)}`))
                            : ''));
                },
                nl,
                (tab) => `View${(0, printTree_1.printTree)(tab, [(tab) => String(JSON.stringify(this.view(), null, 2)).replace(/\n/g, '\n' + tab)])}`,
                nl,
                (tab) => this.clock.toString(tab),
                hasExtensions ? nl : null,
                hasExtensions ? (tab) => this.ext.toString(tab) : null,
            ]));
    }
}
exports.Model = Model;
