"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Decoder = void 0;
const op_1 = require("../../op");
const MsgPackDecoderFast_1 = require("../../../json-pack/msgpack/MsgPackDecoderFast");
const util_1 = require("../../util");
class Decoder extends MsgPackDecoderFast_1.MsgPackDecoderFast {
    constructor(options) {
        super();
        this.options = options;
    }
    decode(uint8) {
        this.reader.reset(uint8);
        return this.decodePatch();
    }
    decodePatch() {
        const len = this.decodeArrayHeader();
        const ops = [];
        for (let i = 0; i < len; i++)
            ops.push(this.decodeOp(undefined));
        return ops;
    }
    decodeOp(parent) {
        const length = this.decodeArrayHeader();
        const opcode = this.reader.u8();
        switch (opcode) {
            case 0: {
                const path = this.decodeArray();
                const value = this.val();
                return new op_1.OpAdd(path, value);
            }
            case 43: {
                const path = this.decodePath(parent);
                const length = this.decodeArrayHeader();
                const ops = [];
                const op = new op_1.OpAnd(path, ops);
                for (let i = 0; i < length; i++)
                    ops.push(this.decodeOp(op));
                return op;
            }
            case 30: {
                const path = this.decodePath(parent);
                const value = this.decodeString();
                const ignoreCase = length > 3;
                return new op_1.OpContains(path, value, ignoreCase);
            }
            case 3: {
                const path = this.decodeArray();
                const from = this.decodeArray();
                return new op_1.OpCopy(path, from);
            }
            case 31: {
                const path = this.decodePath(parent);
                return new op_1.OpDefined(path);
            }
            case 32: {
                const path = this.decodePath(parent);
                const value = this.decodeString();
                const ignoreCase = length > 3;
                return new op_1.OpEnds(path, value, ignoreCase);
            }
            case 12: {
                const path = this.decodeArray();
                const props = this.decodeObject();
                const deleteNull = length > 3;
                return new op_1.OpExtend(path, props, deleteNull);
            }
            case 8: {
                const path = this.decodeArray();
                return new op_1.OpFlip(path);
            }
            case 33: {
                const path = this.decodePath(parent);
                const value = this.decodeArray();
                return new op_1.OpIn(path, value);
            }
            case 9: {
                const path = this.decodePath(parent);
                const inc = this.val();
                return new op_1.OpInc(path, inc);
            }
            case 34: {
                const path = this.decodePath(parent);
                const value = this.val();
                return new op_1.OpLess(path, value);
            }
            case 35: {
                const path = this.decodePath(parent);
                const value = this.decodeString();
                const ignoreCase = length > 3;
                return new op_1.OpMatches(path, value, ignoreCase, this.options.createMatcher || util_1.createMatcherDefault);
            }
            case 11: {
                const hasProps = length > 3;
                const path = this.decodeArray();
                const pos = this.val();
                const props = hasProps ? this.decodeObject() : null;
                return new op_1.OpMerge(path, pos, props);
            }
            case 36: {
                const path = this.decodePath(parent);
                const value = this.val();
                return new op_1.OpMore(path, value);
            }
            case 4: {
                const path = this.decodeArray();
                const from = this.decodeArray();
                return new op_1.OpMove(path, from);
            }
            case 44: {
                const path = this.decodePath(parent);
                const length = this.decodeArrayHeader();
                const ops = [];
                const op = new op_1.OpNot(path, ops);
                for (let i = 0; i < length; i++)
                    ops.push(this.decodeOp(op));
                return op;
            }
            case 45: {
                const path = this.decodePath(parent);
                const length = this.decodeArrayHeader();
                const ops = [];
                const op = new op_1.OpOr(path, ops);
                for (let i = 0; i < length; i++)
                    ops.push(this.decodeOp(op));
                return op;
            }
            case 1: {
                const path = this.decodeArray();
                const hasOldValue = length > 2;
                const oldValue = hasOldValue ? this.val() : undefined;
                return new op_1.OpRemove(path, oldValue);
            }
            case 2: {
                const path = this.decodeArray();
                const value = this.val();
                const hasOldValue = length > 3;
                const oldValue = hasOldValue ? this.val() : undefined;
                return new op_1.OpReplace(path, value, oldValue);
            }
            case 10: {
                const path = this.decodeArray();
                const pos = this.val();
                const hasProps = length > 3;
                const props = hasProps ? this.decodeObject() : null;
                return new op_1.OpSplit(path, pos, props);
            }
            case 37: {
                const ignoreCase = length > 3;
                const path = this.decodePath(parent);
                const value = this.decodeString();
                return new op_1.OpStarts(path, value, ignoreCase);
            }
            case 7: {
                const hasStr = length < 5;
                const path = this.decodeArray();
                const pos = this.val();
                if (hasStr) {
                    const str = this.decodeString();
                    return new op_1.OpStrDel(path, pos, str, undefined);
                }
                else {
                    this.reader.u8();
                    const len = this.val();
                    return new op_1.OpStrDel(path, pos, undefined, len);
                }
            }
            case 6: {
                const path = this.decodeArray();
                const pos = this.val();
                const str = this.decodeString();
                return new op_1.OpStrIns(path, pos, str);
            }
            case 5: {
                const not = length > 3;
                const path = this.decodePath(parent);
                const value = this.val();
                return new op_1.OpTest(path, value, not);
            }
            case 40: {
                const not = length > 4;
                const path = this.decodePath(parent);
                const pos = this.val();
                const str = this.decodeString();
                return new op_1.OpTestString(path, pos, str, not);
            }
            case 41: {
                const not = length > 3;
                const path = this.decodePath(parent);
                const len = this.val();
                return new op_1.OpTestStringLen(path, len, not);
            }
            case 39: {
                const path = this.decodePath(parent);
                const type = this.decodeArray();
                return new op_1.OpTestType(path, type);
            }
            case 42: {
                const path = this.decodePath(parent);
                const value = this.decodeString();
                return new op_1.OpType(path, value);
            }
            case 38: {
                const path = this.decodePath(parent);
                return new op_1.OpUndefined(path);
            }
        }
        throw new Error('OP_UNKNOWN');
    }
    decodePath(parent) {
        const path = this.decodeArray();
        if (!parent)
            return path;
        return [...parent.path, ...path];
    }
    decodeObject() {
        const reader = this.reader;
        const byte = reader.u8();
        if (byte <= 0xbf)
            return this.obj(byte & 0b1111);
        else if (byte === 0xde)
            return this.obj(reader.u16());
        else
            return this.obj(reader.u32());
    }
    decodeArray() {
        const reader = this.reader;
        const byte = reader.u8();
        if (byte < 0b10011111)
            return this.arr(byte & 0b1111);
        else if (byte === 0xdc)
            return this.arr(reader.u16());
        else
            return this.arr(reader.u32());
    }
    decodeArrayHeader() {
        const reader = this.reader;
        const byte = reader.u8();
        if (byte < 0b10011111)
            return byte & 0b1111;
        else if (byte === 0xdc)
            return reader.u16();
        else
            return reader.u32();
    }
    decodeString() {
        const reader = this.reader;
        const byte = reader.u8();
        if (byte <= 0xbf)
            return reader.utf8(byte & 0b11111);
        else if (byte === 0xd9)
            return reader.utf8(reader.u8());
        else if (byte === 0xda)
            return reader.utf8(reader.u16());
        else
            return reader.utf8(reader.u32());
    }
}
exports.Decoder = Decoder;
