"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidatorCodegenContext = void 0;
const codegen_1 = require("../../../util/codegen");
const constants_1 = require("../../constants");
class ValidatorCodegenContext {
    constructor(options) {
        this.validatorCache = new Map();
        this.options = Object.assign({ skipObjectExtraFieldsCheck: false, unsafeMode: false }, options);
        const successResult = this.options.errors === 'boolean' ? 'false' : this.options.errors === 'string' ? "''" : 'null';
        this.codegen = new codegen_1.Codegen({
            epilogue: `return ${successResult};`,
        });
    }
    js(js) {
        this.codegen.js(js);
    }
    err(code, path, opts = {}) {
        switch (this.options.errors) {
            case 'boolean':
                return 'true';
            case 'string': {
                let out = "'[" + JSON.stringify(constants_1.ValidationError[code]);
                for (const step of path) {
                    if (typeof step === 'object') {
                        out += ",' + JSON.stringify(" + step.r + ") + '";
                    }
                    else {
                        out += ',' + JSON.stringify(step);
                    }
                }
                return out + "]'";
            }
            case 'object':
            default: {
                let out = '{code: ' +
                    JSON.stringify(constants_1.ValidationError[code]) +
                    ', errno: ' +
                    JSON.stringify(code) +
                    ', message: ' +
                    JSON.stringify(constants_1.ValidationErrorMessage[code]) +
                    ', path: [';
                let i = 0;
                for (const step of path) {
                    if (i)
                        out += ', ';
                    if (typeof step === 'object') {
                        out += step.r;
                    }
                    else {
                        out += JSON.stringify(step);
                    }
                    i++;
                }
                out += ']';
                if (opts.refId) {
                    out += ', refId: ' + JSON.stringify(opts.refId);
                }
                if (opts.refError) {
                    out += ', ref: ' + opts.refError;
                }
                if (opts.validator) {
                    out += ', validator: ' + JSON.stringify(opts.validator);
                }
                return out + '}';
            }
        }
    }
    emitCustomValidators(node, path, r) {
        const validatorNames = node.getValidatorNames();
        for (const validatorName of validatorNames) {
            const codegen = this.codegen;
            const v = this.linkValidator(validatorName);
            const rerr = codegen.getRegister();
            const rc = codegen.getRegister();
            const err = this.err(constants_1.ValidationError.VALIDATION, path, { validator: validatorName, refError: rerr });
            const errInCatch = this.err(constants_1.ValidationError.VALIDATION, path, { validator: validatorName, refError: rc });
            const emitRc = this.options.errors === 'object';
            codegen.js(`try {
  var ${rerr} = ${v}(${r});
  if (${rerr}) return ${err};
} catch (e) {
  ${emitRc ? `var ${rc} = e ? e : new Error('Validator ${JSON.stringify(validatorName)} failed.');` : ''}
  return ${errInCatch};
}`);
        }
    }
    linkValidator(name) {
        const cached = this.validatorCache.get(name);
        if (cached)
            return cached;
        const system = this.options.system;
        if (!system)
            throw new Error('Type system not set.');
        const codegen = this.codegen;
        const validator = system.getCustomValidator(name);
        if (!validator)
            throw new Error(`Custom validator "${name}" not found.`);
        const result = codegen.linkDependency(validator.fn);
        this.validatorCache.set(name, result);
        return result;
    }
    compile() {
        return this.codegen.compile();
    }
}
exports.ValidatorCodegenContext = ValidatorCodegenContext;
