"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeSystem = void 0;
const TypeAlias_1 = require("./TypeAlias");
const TypeBuilder_1 = require("../type/TypeBuilder");
const classes_1 = require("../type/classes");
const printTree_1 = require("../../util/print/printTree");
class TypeSystem {
    constructor() {
        this.t = new TypeBuilder_1.TypeBuilder(this);
        this.aliases = new Map();
        this.alias = (id, type) => {
            const existingAlias = this.aliases.get(id);
            if (existingAlias)
                return existingAlias;
            const alias = new TypeAlias_1.TypeAlias(this, id, type);
            this.aliases.set(id, alias);
            return alias;
        };
        this.unalias = (id) => {
            const alias = this.aliases.get(id);
            if (!alias)
                throw new Error(`Alias [id = ${id}] not found.`);
            return alias;
        };
        this.hasAlias = (id) => this.aliases.has(id);
        this.resolve = (id) => {
            const alias = this.unalias(id);
            return alias.type instanceof classes_1.RefType ? this.resolve(alias.type.getRef()) : alias;
        };
        this.customValidators = new Map();
        this.addCustomValidator = (validator) => {
            if (this.customValidators.has(validator.name))
                throw new Error(`Validator [name = ${validator.name}] already exists.`);
            this.customValidators.set(validator.name, validator);
        };
        this.getCustomValidator = (name) => {
            const validator = this.customValidators.get(name);
            if (!validator)
                throw new Error(`Validator [name = ${name}] not found.`);
            return validator;
        };
    }
    importTypes(types) {
        const result = {};
        for (const id in types)
            result[id] = this.alias(id, types[id]);
        return result;
    }
    exportTypes() {
        const result = {};
        for (const [id, alias] of this.aliases.entries()) {
            result[id] = alias.getType().getSchema();
        }
        return result;
    }
    toString(tab = '') {
        const nl = () => '';
        return (this.constructor.name +
            (0, printTree_1.printTree)(tab, [
                (tab) => 'aliases' +
                    (0, printTree_1.printTree)(tab, [...this.aliases.values()].map((alias) => (tab) => alias.toString(tab))),
                this.customValidators.size ? nl : null,
                (tab) => 'validators' +
                    (0, printTree_1.printTree)(tab, [...this.customValidators.keys()].map((validator) => (tab) => `"${validator}"`)),
            ]));
    }
}
exports.TypeSystem = TypeSystem;
