"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RpcMessageBatchProcessor = void 0;
const tslib_1 = require("tslib");
const msg = tslib_1.__importStar(require("../messages"));
const caller_1 = require("./caller");
const validation_1 = require("./validation");
class RpcMessageBatchProcessor {
    constructor({ caller }) {
        this.caller = caller;
    }
    onBatch(list, ctx) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const promises = [];
                const length = list.length;
                for (let i = 0; i < length; i++) {
                    const message = list[i];
                    switch (message.constructor) {
                        case msg.NotificationMessage:
                            this.onNotification(message, ctx);
                            break;
                        case msg.RequestDataMessage:
                        case msg.RequestCompleteMessage:
                        case msg.RequestErrorMessage:
                            promises.push(this.onRequest(message, ctx));
                            break;
                    }
                }
                const settled = yield Promise.allSettled(promises);
                const result = [];
                const settledLength = settled.length;
                for (let i = 0; i < settledLength; i++) {
                    const item = settled[i];
                    result.push(item.status === 'fulfilled' ? item.value : item.reason);
                }
                return result;
            }
            catch (error) {
                const value = caller_1.RpcError.internalErrorValue();
                return [new msg.ResponseErrorMessage(-1, value)];
            }
        });
    }
    onNotification(message, ctx) {
        const method = message.method;
        (0, validation_1.validateMethod)(method);
        this.caller.notification(method, message.value.data, ctx).catch((error) => { });
    }
    onRequest(message, ctx) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const id = message.id;
            (0, validation_1.validateId)(id);
            const method = message.method;
            (0, validation_1.validateMethod)(method);
            try {
                const value = message.value;
                const data = value ? value.data : undefined;
                const result = yield this.caller.call(method, data, ctx);
                return new msg.ResponseCompleteMessage(id, result);
            }
            catch (error) {
                throw new msg.ResponseErrorMessage(id, error);
            }
        });
    }
}
exports.RpcMessageBatchProcessor = RpcMessageBatchProcessor;
