"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RpcCaller = void 0;
const tslib_1 = require("tslib");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const BufferSubject_1 = require("../../../../util/rx/BufferSubject");
const error_1 = require("./error");
const Value_1 = require("../../messages/Value");
const StaticRpcMethod_1 = require("../methods/StaticRpcMethod");
const INVALID_REQUEST_ERROR_VALUE = error_1.RpcError.value(error_1.RpcError.invalidRequest());
const defaultWrapInternalError = (error) => error_1.RpcError.valueFrom(error);
class RpcCaller {
    constructor({ getMethod, preCallBufferSize = 10, wrapInternalError = defaultWrapInternalError, }) {
        this.getMethod = getMethod;
        this.preCallBufferSize = preCallBufferSize;
        this.wrapInternalError = wrapInternalError;
    }
    exists(name) {
        return !!this.getMethod(name);
    }
    getMethodStrict(name) {
        const method = this.getMethod(name);
        if (!method)
            throw error_1.RpcError.valueFromCode(error_1.RpcErrorCodes.METHOD_NOT_FOUND);
        return method;
    }
    info(name) {
        return this.getMethodStrict(name);
    }
    validate(method, request) {
        const validate = method.validate;
        if (!validate)
            return;
        try {
            const errors = validate(request);
            if (errors)
                throw errors;
        }
        catch (error) {
            throw this.wrapValidationError(error);
        }
    }
    wrapValidationError(error) {
        return error_1.RpcError.valueFrom(error, INVALID_REQUEST_ERROR_VALUE);
    }
    call(name, request, ctx) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const method = this.getMethodStrict(name);
                this.validate(method, request);
                const preCall = method.onPreCall;
                if (preCall)
                    yield preCall(ctx, request);
                const data = yield method.call(request, ctx);
                return new Value_1.Value(data, method.res);
            }
            catch (error) {
                throw this.wrapInternalError(error);
            }
        });
    }
    notification(name, request, ctx) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const method = this.getMethodStrict(name);
            if (!(method instanceof StaticRpcMethod_1.StaticRpcMethod))
                return;
            if (!method.acceptsNotifications)
                return;
            this.validate(method, request);
            try {
                if (method.onPreCall)
                    yield method.onPreCall(ctx, request);
                yield method.call(request, ctx);
            }
            catch (error) {
                throw this.wrapInternalError(error);
            }
        });
    }
    createCall(name, ctx) {
        const req$ = new rxjs_1.Subject();
        const reqUnsubscribe$ = new rxjs_1.Subject();
        try {
            const method = this.getMethodStrict(name);
            if (!method.isStreaming) {
                const response$ = (0, rxjs_1.from)((() => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const request = yield (0, rxjs_1.firstValueFrom)(req$.pipe((0, operators_1.first)()));
                    return yield this.call(name, request, ctx);
                }))());
                const res$ = new rxjs_1.Subject();
                response$.subscribe(res$);
                const $resWithErrorsFormatted = res$.pipe((0, operators_1.catchError)((error) => {
                    throw this.wrapInternalError(error);
                }));
                return { req$, reqUnsubscribe$, res$: $resWithErrorsFormatted };
            }
            const methodStreaming = method;
            const requestValidated$ = req$.pipe((0, operators_1.tap)((request) => {
                this.validate(methodStreaming, request);
            }));
            const bufferSize = methodStreaming.preCallBufferSize || this.preCallBufferSize;
            const requestBuffered$ = new BufferSubject_1.BufferSubject(bufferSize);
            const error$ = new rxjs_1.Subject();
            requestBuffered$.subscribe({
                error: (error) => {
                    error$.error(error);
                },
            });
            requestValidated$.subscribe(requestBuffered$);
            const methodResponseType = method.res;
            const result$ = requestBuffered$.pipe((0, operators_1.take)(1), (0, operators_1.switchMap)((request) => {
                return methodStreaming.onPreCall ? (0, rxjs_1.from)(methodStreaming.onPreCall(ctx, request)) : (0, rxjs_1.from)([0]);
            }), (0, operators_1.switchMap)(() => {
                Promise.resolve().then(() => {
                    requestBuffered$.flush();
                });
                return method.call$(requestBuffered$, ctx).pipe((0, operators_1.map)((response) => new Value_1.Value(response, methodResponseType)), (0, operators_1.finalize)(() => {
                    error$.complete();
                }));
            }), (0, operators_1.share)(), (0, operators_1.mergeWith)(error$));
            const $resWithErrorsFormatted = result$.pipe((0, operators_1.finalize)(() => {
                error$.complete();
            }), (0, operators_1.catchError)((error) => {
                throw error_1.RpcError.valueFrom(error);
            }));
            return { req$, reqUnsubscribe$, res$: $resWithErrorsFormatted };
        }
        catch (error) {
            const errorFormatted = error_1.RpcError.valueFrom(error);
            req$.error(errorFormatted);
            const res$ = new rxjs_1.Subject();
            res$.error(errorFormatted);
            return { req$, reqUnsubscribe$, res$ };
        }
    }
    call$(name, request$, ctx) {
        const call = this.createCall(name, ctx);
        request$.subscribe(call.req$);
        return call.res$;
    }
}
exports.RpcCaller = RpcCaller;
