import { NewConOp, NewObjOp, NewValOp, NewVecOp, NewStrOp, NewBinOp, NewArrOp, InsValOp, InsObjOp, InsVecOp, InsStrOp, InsBinOp, InsArrOp, DelOp, NopOp, } from './operations';
import { ts, Timestamp } from './clock';
import { isUint8Array } from '../util/buffers/isUint8Array';
import { Patch } from './Patch';
import { ORIGIN } from './constants';
import { VectorDelayedValue } from './builder/Tuple';
import { Konst } from './builder/Konst';
import { NodeBuilder } from './builder/DelayedValueBuilder';
const maybeConst = (x) => {
    switch (typeof x) {
        case 'number':
        case 'boolean':
            return true;
        default:
            return x === null;
    }
};
export class PatchBuilder {
    clock;
    patch;
    constructor(clock) {
        this.clock = clock;
        this.patch = new Patch();
    }
    nextTime() {
        return this.patch.nextTime() || this.clock.time;
    }
    flush() {
        const patch = this.patch;
        this.patch = new Patch();
        return patch;
    }
    obj() {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new NewObjOp(id));
        return id;
    }
    arr() {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new NewArrOp(id));
        return id;
    }
    vec() {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new NewVecOp(id));
        return id;
    }
    str() {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new NewStrOp(id));
        return id;
    }
    bin() {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new NewBinOp(id));
        return id;
    }
    const(value) {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new NewConOp(id, value));
        return id;
    }
    val() {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new NewValOp(id));
        return id;
    }
    root(val) {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new InsValOp(id, ORIGIN, val));
        return id;
    }
    insObj(obj, data) {
        this.pad();
        if (!data.length)
            throw new Error('EMPTY_TUPLES');
        const id = this.clock.tick(1);
        const op = new InsObjOp(id, obj, data);
        const span = op.span();
        if (span > 1)
            this.clock.tick(span - 1);
        this.patch.ops.push(op);
        return id;
    }
    insVec(obj, data) {
        this.pad();
        if (!data.length)
            throw new Error('EMPTY_TUPLES');
        const id = this.clock.tick(1);
        const op = new InsVecOp(id, obj, data);
        const span = op.span();
        if (span > 1)
            this.clock.tick(span - 1);
        this.patch.ops.push(op);
        return id;
    }
    setVal(obj, val) {
        this.pad();
        const id = this.clock.tick(1);
        const op = new InsValOp(id, obj, val);
        this.patch.ops.push(op);
        return id;
    }
    insStr(obj, ref, data) {
        this.pad();
        if (!data.length)
            throw new Error('EMPTY_STRING');
        const id = this.clock.tick(1);
        const op = new InsStrOp(id, obj, ref, data);
        const span = op.span();
        if (span > 1)
            this.clock.tick(span - 1);
        this.patch.ops.push(op);
        return id;
    }
    insBin(obj, ref, data) {
        this.pad();
        if (!data.length)
            throw new Error('EMPTY_BINARY');
        const id = this.clock.tick(1);
        const op = new InsBinOp(id, obj, ref, data);
        const span = op.span();
        if (span > 1)
            this.clock.tick(span - 1);
        this.patch.ops.push(op);
        return id;
    }
    insArr(arr, ref, data) {
        this.pad();
        const id = this.clock.tick(1);
        const op = new InsArrOp(id, arr, ref, data);
        const span = op.span();
        if (span > 1)
            this.clock.tick(span - 1);
        this.patch.ops.push(op);
        return id;
    }
    del(obj, what) {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new DelOp(id, obj, what));
        return id;
    }
    nop(span) {
        this.pad();
        const id = this.clock.tick(span);
        this.patch.ops.push(new NopOp(id, span));
        return id;
    }
    jsonObj(obj) {
        const id = this.obj();
        const keys = Object.keys(obj);
        if (keys.length) {
            const tuples = [];
            for (const k of keys) {
                const value = obj[k];
                const valueId = value instanceof Timestamp ? value : maybeConst(value) ? this.const(value) : this.json(value);
                tuples.push([k, valueId]);
            }
            this.insObj(id, tuples);
        }
        return id;
    }
    jsonArr(arr) {
        const id = this.arr();
        if (arr.length) {
            const values = [];
            for (const el of arr)
                values.push(this.json(el));
            this.insArr(id, id, values);
        }
        return id;
    }
    jsonStr(str) {
        const id = this.str();
        if (str)
            this.insStr(id, id, str);
        return id;
    }
    jsonBin(bin) {
        const id = this.bin();
        if (bin.length)
            this.insBin(id, id, bin);
        return id;
    }
    jsonVal(value) {
        const valId = this.val();
        const id = this.const(value);
        this.setVal(valId, id);
        return valId;
    }
    jsonVec(vector) {
        const id = this.vec();
        const length = vector.length;
        if (length) {
            const writes = [];
            for (let i = 0; i < length; i++)
                writes.push([i, this.constOrJson(vector[i])]);
            this.insVec(id, writes);
        }
        return id;
    }
    json(json) {
        if (json instanceof Timestamp)
            return json;
        if (json === undefined)
            return this.const(json);
        if (json instanceof Array)
            return this.jsonArr(json);
        if (isUint8Array(json))
            return this.jsonBin(json);
        if (json instanceof VectorDelayedValue)
            return this.jsonVec(json.slots);
        if (json instanceof Konst)
            return this.const(json.val);
        if (json instanceof NodeBuilder)
            return json.build(this);
        switch (typeof json) {
            case 'object':
                return json === null ? this.jsonVal(json) : this.jsonObj(json);
            case 'string':
                return this.jsonStr(json);
            case 'number':
            case 'boolean':
                return this.jsonVal(json);
        }
        throw new Error('INVALID_JSON');
    }
    constOrJson(value) {
        if (value instanceof Timestamp)
            return value;
        return maybeConst(value) ? this.const(value) : this.json(value);
    }
    maybeConst(value) {
        return value instanceof Timestamp ? value : this.const(value);
    }
    pad() {
        const nextTime = this.patch.nextTime();
        if (!nextTime)
            return;
        const drift = this.clock.time - nextTime;
        if (drift > 0) {
            const id = ts(this.clock.sid, nextTime);
            const padding = new NopOp(id, drift);
            this.patch.ops.push(padding);
        }
    }
}
