import { Expression, Literal } from '../codegen-steps';
import { deepEqual } from '../../json-equal/deepEqual';
import { $$deepEqual } from '../../json-equal/$$deepEqual';
import * as util from '../util';
const eqLitVsExpr = (literal, expression, ctx, not) => {
    const fn = $$deepEqual(literal.val);
    const d = ctx.const(fn);
    return new Expression(`${not ? '!' : ''}${d}(${expression})`);
};
const binaryOperands = (expr, ctx) => {
    const left = ctx.eval(expr[1], ctx);
    const right = ctx.eval(expr[2], ctx);
    return [left, right];
};
const ternaryOperands = (expr, ctx) => {
    const a = ctx.eval(expr[1], ctx);
    const b = ctx.eval(expr[2], ctx);
    const c = ctx.eval(expr[3], ctx);
    return [a, b, c];
};
export const comparisonOperators = [
    [
        '==',
        ['eq'],
        2,
        (expr, ctx) => {
            const [left, right] = binaryOperands(expr, ctx);
            return deepEqual(left, right);
        },
        (ctx) => {
            const a = ctx.operands[0];
            const b = ctx.operands[1];
            if (a instanceof Literal && b instanceof Expression)
                return eqLitVsExpr(a, b, ctx);
            if (b instanceof Literal && a instanceof Expression)
                return eqLitVsExpr(b, a, ctx);
            ctx.link(deepEqual, 'deepEqual');
            return new Expression(`deepEqual(${a},${b})`);
        },
    ],
    [
        '!=',
        ['ne'],
        2,
        (expr, ctx) => {
            const [left, right] = binaryOperands(expr, ctx);
            return !deepEqual(left, right);
        },
        (ctx) => {
            const a = ctx.operands[0];
            const b = ctx.operands[1];
            if (a instanceof Literal && b instanceof Expression)
                return eqLitVsExpr(a, b, ctx, true);
            if (b instanceof Literal && a instanceof Expression)
                return eqLitVsExpr(b, a, ctx, true);
            ctx.link(deepEqual, 'deepEqual');
            return new Expression(`!deepEqual(${a},${b})`);
        },
    ],
    [
        '>',
        ['gt'],
        2,
        (expr, ctx) => {
            const [left, right] = binaryOperands(expr, ctx);
            return left > right;
        },
        (ctx) => {
            return new Expression(`(${ctx.operands[0]})>(${ctx.operands[1]})`);
        },
    ],
    [
        '>=',
        ['ge'],
        2,
        (expr, ctx) => {
            const [left, right] = binaryOperands(expr, ctx);
            return left >= right;
        },
        (ctx) => {
            return new Expression(`(${ctx.operands[0]})>=(${ctx.operands[1]})`);
        },
    ],
    [
        '<',
        ['lt'],
        2,
        (expr, ctx) => {
            const [left, right] = binaryOperands(expr, ctx);
            return left < right;
        },
        (ctx) => {
            return new Expression(`(${ctx.operands[0]})<(${ctx.operands[1]})`);
        },
    ],
    [
        '<=',
        ['le'],
        2,
        (expr, ctx) => {
            const [left, right] = binaryOperands(expr, ctx);
            return left <= right;
        },
        (ctx) => {
            return new Expression(`(${ctx.operands[0]})<=(${ctx.operands[1]})`);
        },
    ],
    [
        'cmp',
        [],
        2,
        (expr, ctx) => {
            const [left, right] = binaryOperands(expr, ctx);
            return util.cmp(left, right);
        },
        (ctx) => {
            ctx.link(util.cmp, 'cmp');
            return new Expression(`cmp((${ctx.operands[0]}),(${ctx.operands[1]}))`);
        },
    ],
    [
        '=><=',
        ['between'],
        3,
        (expr, ctx) => {
            const [val, min, max] = ternaryOperands(expr, ctx);
            return util.betweenEqEq(val, min, max);
        },
        (ctx) => {
            ctx.link(util.betweenEqEq, 'betweenEqEq');
            return new Expression(`betweenEqEq(${ctx.operands[0]},${ctx.operands[1]},${ctx.operands[2]})`);
        },
    ],
    [
        '><',
        [],
        3,
        (expr, ctx) => {
            const [val, min, max] = ternaryOperands(expr, ctx);
            return util.betweenNeNe(val, min, max);
        },
        (ctx) => {
            ctx.link(util.betweenNeNe, 'betweenNeNe');
            return new Expression(`betweenNeNe(${ctx.operands[0]},${ctx.operands[1]},${ctx.operands[2]})`);
        },
    ],
    [
        '=><',
        [],
        3,
        (expr, ctx) => {
            const [val, min, max] = ternaryOperands(expr, ctx);
            return util.betweenEqNe(val, min, max);
        },
        (ctx) => {
            ctx.link(util.betweenEqNe, 'betweenEqNe');
            return new Expression(`betweenEqNe(${ctx.operands[0]},${ctx.operands[1]},${ctx.operands[2]})`);
        },
    ],
    [
        '><=',
        [],
        3,
        (expr, ctx) => {
            const [val, min, max] = ternaryOperands(expr, ctx);
            return util.betweenNeEq(val, min, max);
        },
        (ctx) => {
            ctx.link(util.betweenNeEq, 'betweenNeEq');
            return new Expression(`betweenNeEq(${ctx.operands[0]},${ctx.operands[1]},${ctx.operands[2]})`);
        },
    ],
];
