import { Expression, Literal } from '../codegen-steps';
import * as util from '../util';
const binaryOperands = (expr, ctx) => {
    const left = ctx.eval(expr[1], ctx);
    const right = ctx.eval(expr[2], ctx);
    return [left, right];
};
const createValidationOperator = (name, validate) => {
    return [
        name + '?',
        [],
        1,
        (expr, ctx) => {
            const email = ctx.eval(expr[1], ctx);
            return validate(email);
        },
        (ctx) => {
            ctx.link(validate, 'is_' + name);
            return new Expression(`is_${name}(${ctx.operands[0]})`);
        },
    ];
};
export const stringOperators = [
    [
        '.',
        ['cat'],
        -1,
        (expr, ctx) => {
            return expr.slice(1).reduce((acc, e) => acc + util.str(ctx.eval(e, ctx)), '');
        },
        (ctx) => {
            ctx.link(util.str, 'str');
            const js = ctx.operands.map((expr) => `str(${expr})`).join('+');
            return new Expression(js);
        },
    ],
    [
        'contains',
        [],
        2,
        (expr, ctx) => {
            const [outer, inner] = binaryOperands(expr, ctx);
            return util.contains(outer, inner);
        },
        (ctx) => {
            ctx.link(util.contains, 'contains');
            const js = `contains(${ctx.operands[0]},${ctx.operands[1]})`;
            return new Expression(js);
        },
    ],
    [
        'starts',
        [],
        2,
        (expr, ctx) => {
            const [outer, inner] = binaryOperands(expr, ctx);
            return util.starts(outer, inner);
        },
        (ctx) => {
            ctx.link(util.starts, 'starts');
            const js = `starts(${ctx.operands[0]},${ctx.operands[1]})`;
            return new Expression(js);
        },
    ],
    [
        'ends',
        [],
        2,
        (expr, ctx) => {
            const [outer, inner] = binaryOperands(expr, ctx);
            return util.ends(outer, inner);
        },
        (ctx) => {
            ctx.link(util.ends, 'ends');
            const js = `ends(${ctx.operands[0]},${ctx.operands[1]})`;
            return new Expression(js);
        },
    ],
    [
        'substr',
        [],
        3,
        (expr, ctx) => {
            const str = ctx.eval(expr[1], ctx);
            const start = ctx.eval(expr[2], ctx);
            const end = ctx.eval(expr[3], ctx);
            return util.substr(str, start, end);
        },
        (ctx) => {
            ctx.link(util.substr, 'substr');
            const js = `substr(${ctx.operands[0]},${ctx.operands[1]},${ctx.operands[2]})`;
            return new Expression(js);
        },
    ],
    [
        'matches',
        [],
        2,
        (expr, ctx) => {
            let pattern = expr[2];
            if (pattern instanceof Array && pattern.length === 1)
                pattern = pattern[0];
            if (typeof pattern !== 'string')
                throw new Error('"matches" second argument should be a regular expression string.');
            if (!ctx.createPattern)
                throw new Error('"matches" operator requires ".createPattern()" option to be implemented.');
            const fn = ctx.createPattern(pattern);
            const outer = ctx.eval(expr[1], ctx);
            return fn(util.str(outer));
        },
        (ctx) => {
            const pattern = ctx.operands[1];
            if (!(pattern instanceof Literal) || typeof pattern.val !== 'string')
                throw new Error('"matches" second argument should be a regular expression string.');
            if (!ctx.createPattern)
                throw new Error('"matches" operator requires ".createPattern()" option to be implemented.');
            const fn = ctx.createPattern(pattern.val);
            const d = ctx.link(fn);
            ctx.link(util.str, 'str');
            const subject = ctx.operands[0];
            return new Expression(`${d}(str(${subject}))`);
        },
    ],
    createValidationOperator('email', util.isEmail),
    createValidationOperator('hostname', util.isHostname),
    createValidationOperator('ip4', util.isIp4),
    createValidationOperator('ip6', util.isIp6),
    createValidationOperator('uuid', util.isUuid),
    createValidationOperator('uri', util.isUri),
    createValidationOperator('duration', util.isDuration),
    createValidationOperator('date', util.isDate),
    createValidationOperator('time', util.isTime),
    createValidationOperator('dateTime', util.isDateTime),
];
