import { isFloat32 } from '../../util/buffers/isFloat32';
import { JsonPackExtension } from '../JsonPackExtension';
import { CborEncoderFast } from './CborEncoderFast';
const isSafeInteger = Number.isSafeInteger;
export class CborEncoder extends CborEncoderFast {
    writeUnknown(value) {
        this.writeNull();
    }
    writeAny(value) {
        switch (typeof value) {
            case 'number':
                return this.writeNumber(value);
            case 'string':
                return this.writeStr(value);
            case 'boolean':
                return this.writer.u8(0xf4 + +value);
            case 'object': {
                if (!value)
                    return this.writer.u8(0xf6);
                const constructor = value.constructor;
                switch (constructor) {
                    case Object:
                        return this.writeObj(value);
                    case Array:
                        return this.writeArr(value);
                    case Uint8Array:
                        return this.writeBin(value);
                    case Map:
                        return this.writeMap(value);
                    case JsonPackExtension:
                        return this.writeTag(value.tag, value.val);
                    default:
                        return this.writeUnknown(value);
                }
            }
            case 'undefined':
                return this.writeUndef();
            case 'bigint':
                return this.writeBigInt(value);
            default:
                return this.writeUnknown(value);
        }
    }
    writeNumber(num) {
        if (isSafeInteger(num))
            this.writeInteger(num);
        else if (typeof num === 'bigint')
            this.writeBigInt(num);
        else
            this.writeFloat(num);
    }
    writeBigInt(int) {
        if (int >= 0)
            this.writeBigUint(int);
        else
            this.writeBigSint(int);
    }
    writeBigUint(uint) {
        if (uint <= Number.MAX_SAFE_INTEGER)
            return this.writeUInteger(Number(uint));
        this.writer.u8u64(0x1b, uint);
    }
    writeBigSint(int) {
        if (int >= Number.MIN_SAFE_INTEGER)
            return this.encodeNint(Number(int));
        const uint = -BigInt(1) - int;
        this.writer.u8u64(0x3b, uint);
    }
    writeFloat(float) {
        if (isFloat32(float))
            this.writer.u8f32(0xfa, float);
        else
            this.writer.u8f64(0xfb, float);
    }
    writeMap(map) {
        this.writeMapHdr(map.size);
        map.forEach((value, key) => {
            this.writeAny(key);
            this.writeAny(value);
        });
    }
    writeUndef() {
        this.writer.u8(0xf7);
    }
}
