import { Writer } from '../../util/buffers/Writer';
export class CborEncoderFast {
    writer;
    constructor(writer = new Writer()) {
        this.writer = writer;
    }
    encode(value) {
        this.writeAny(value);
        return this.writer.flush();
    }
    encodeToSlice(value) {
        this.writeAny(value);
        return this.writer.flushSlice();
    }
    writeAny(value) {
        switch (typeof value) {
            case 'number':
                return this.writeNumber(value);
            case 'string':
                return this.writeStr(value);
            case 'boolean':
                return this.writer.u8(0xf4 + +value);
            case 'object': {
                if (!value)
                    return this.writer.u8(0xf6);
                const constructor = value.constructor;
                switch (constructor) {
                    case Array:
                        return this.writeArr(value);
                    default:
                        return this.writeObj(value);
                }
            }
        }
    }
    writeCbor() {
        this.writer.u8u16(0xd9, 0xd9f7);
    }
    writeEnd() {
        this.writer.u8(255);
    }
    writeNull() {
        this.writer.u8(0xf6);
    }
    writeBoolean(bool) {
        if (bool)
            this.writer.u8(0xf5);
        else
            this.writer.u8(0xf4);
    }
    writeNumber(num) {
        if (num >>> 0 === num)
            return this.writeUInteger(num);
        if (num >> 0 === num)
            return this.encodeNint(num);
        this.writeFloat(num);
    }
    writeInteger(int) {
        if (int >= 0)
            this.writeUInteger(int);
        else
            this.encodeNint(int);
    }
    writeUInteger(uint) {
        const writer = this.writer;
        writer.ensureCapacity(9);
        const uint8 = writer.uint8;
        if (uint <= 23) {
            uint8[writer.x++] = 0 + uint;
        }
        else if (uint <= 0xff) {
            uint8[writer.x++] = 0x18;
            uint8[writer.x++] = uint;
        }
        else if (uint <= 0xffff) {
            uint8[writer.x++] = 0x19;
            writer.view.setUint16(writer.x, uint);
            writer.x += 2;
        }
        else if (uint <= 0xffffffff) {
            uint8[writer.x++] = 0x1a;
            writer.view.setUint32(writer.x, uint);
            writer.x += 4;
        }
        else {
            uint8[writer.x++] = 0x1b;
            writer.view.setBigUint64(writer.x, BigInt(uint));
            writer.x += 8;
        }
    }
    encodeNumber(num) {
        this.writeNumber(num);
    }
    encodeInteger(int) {
        this.writeInteger(int);
    }
    encodeUint(uint) {
        this.writeUInteger(uint);
    }
    encodeNint(int) {
        const uint = -1 - int;
        const writer = this.writer;
        writer.ensureCapacity(9);
        const uint8 = writer.uint8;
        if (uint < 24) {
            uint8[writer.x++] = 32 + uint;
        }
        else if (uint <= 0xff) {
            uint8[writer.x++] = 0x38;
            uint8[writer.x++] = uint;
        }
        else if (uint <= 0xffff) {
            uint8[writer.x++] = 0x39;
            writer.view.setUint16(writer.x, uint);
            writer.x += 2;
        }
        else if (uint <= 0xffffffff) {
            uint8[writer.x++] = 0x3a;
            writer.view.setUint32(writer.x, uint);
            writer.x += 4;
        }
        else {
            uint8[writer.x++] = 0x3b;
            writer.view.setBigUint64(writer.x, BigInt(uint));
            writer.x += 8;
        }
    }
    writeFloat(float) {
        this.writer.u8f64(0xfb, float);
    }
    writeBin(buf) {
        const length = buf.length;
        this.writeBinHdr(length);
        this.writer.buf(buf, length);
    }
    writeBinHdr(length) {
        if (length <= 23)
            this.writer.u8(64 + length);
        else if (length <= 0xff)
            this.writer.u16((0x58 << 8) + length);
        else if (length <= 0xffff)
            this.writer.u8u16(0x59, length);
        else if (length <= 0xffffffff)
            this.writer.u8u32(0x5a, length);
        else
            this.writer.u8u64(0x5b, length);
    }
    writeStartBin() {
        this.writer.u8(0x5f);
    }
    writeStr(str) {
        const writer = this.writer;
        const length = str.length;
        const maxSize = length * 4;
        writer.ensureCapacity(5 + maxSize);
        const uint8 = writer.uint8;
        let lengthOffset = writer.x;
        if (maxSize <= 23)
            writer.x++;
        else if (maxSize <= 0xff) {
            uint8[writer.x++] = 0x78;
            lengthOffset = writer.x;
            writer.x++;
        }
        else if (maxSize <= 0xffff) {
            uint8[writer.x++] = 0x79;
            lengthOffset = writer.x;
            writer.x += 2;
        }
        else {
            uint8[writer.x++] = 0x7a;
            lengthOffset = writer.x;
            writer.x += 4;
        }
        const bytesWritten = writer.utf8(str);
        if (maxSize <= 23)
            uint8[lengthOffset] = 96 + bytesWritten;
        else if (maxSize <= 0xff)
            uint8[lengthOffset] = bytesWritten;
        else if (maxSize <= 0xffff)
            writer.view.setUint16(lengthOffset, bytesWritten);
        else
            writer.view.setUint32(lengthOffset, bytesWritten);
    }
    writeStrHdr(length) {
        if (length <= 23)
            this.writer.u8(96 + length);
        else if (length <= 0xff)
            this.writer.u16((0x78 << 8) + length);
        else if (length <= 0xffff)
            this.writer.u8u16(0x79, length);
        else
            this.writer.u8u32(0x7a, length);
    }
    writeAsciiStr(str) {
        this.writeStrHdr(str.length);
        this.writer.ascii(str);
    }
    writeStartStr() {
        this.writer.u8(0x7f);
    }
    writeArr(arr) {
        const length = arr.length;
        this.writeArrHdr(length);
        for (let i = 0; i < length; i++)
            this.writeAny(arr[i]);
    }
    writeArrHdr(length) {
        if (length <= 23)
            this.writer.u8(128 + length);
        else if (length <= 0xff)
            this.writer.u16((0x98 << 8) + length);
        else if (length <= 0xffff)
            this.writer.u8u16(0x99, length);
        else if (length <= 0xffffffff)
            this.writer.u8u32(0x9a, length);
        else
            this.writer.u8u64(0x9b, length);
    }
    writeStartArr() {
        this.writer.u8(0x9f);
    }
    writeEndArr() {
        this.writer.u8(255);
    }
    writeObj(obj) {
        const keys = Object.keys(obj);
        const length = keys.length;
        this.writeObjHdr(length);
        for (let i = 0; i < length; i++) {
            const key = keys[i];
            this.writeStr(key);
            this.writeAny(obj[key]);
        }
    }
    writeObjHdr(length) {
        if (length <= 23)
            this.writer.u8(160 + length);
        else if (length <= 0xff)
            this.writer.u16((0xb8 << 8) + length);
        else if (length <= 0xffff)
            this.writer.u8u16(0xb9, length);
        else if (length <= 0xffffffff)
            this.writer.u8u32(0xba, length);
        else
            this.writer.u8u64(0xbb, length);
    }
    writeMapHdr(length) {
        this.writeObjHdr(length);
    }
    writeStartMap() {
        this.writer.u8(0xbf);
    }
    writeStartObj() {
        this.writer.u8(0xbf);
    }
    writeEndObj() {
        this.writer.u8(255);
    }
    writeTag(tag, value) {
        this.writeTagHdr(tag);
        this.writeAny(value);
    }
    writeTagHdr(tag) {
        if (tag <= 23)
            this.writer.u8(192 + tag);
        else if (tag <= 0xff)
            this.writer.u16((0xd8 << 8) + tag);
        else if (tag <= 0xffff)
            this.writer.u8u16(0xd9, tag);
        else if (tag <= 0xffffffff)
            this.writer.u8u32(0xda, tag);
        else
            this.writer.u8u64(0xdb, tag);
    }
    writeTkn(value) {
        if (value <= 23)
            this.writer.u8(224 + value);
        else if (value <= 0xff)
            this.writer.u16((0xf8 << 8) + value);
    }
}
