import { utf8Count } from '../../util/strings/utf8';
export class NullAstNode {
    val = null;
    len = 1;
    byteLength() {
        return 1;
    }
}
export class BoolAstNode {
    val;
    len = 1;
    constructor(val) {
        this.val = val;
    }
    byteLength() {
        return 1;
    }
}
export class UintAstNode {
    val;
    len;
    constructor(val) {
        this.val = val;
        if (!val)
            this.len = 0;
        else if (val <= 0xff)
            this.len = 1;
        else if (val <= 0xffff)
            this.len = 2;
        else if (val <= 0xffffff)
            this.len = 3;
        else if (val <= 0xffffffff)
            this.len = 4;
        else if (val <= 0xffffffffff)
            this.len = 5;
        else if (val <= 0xffffffffffff)
            this.len = 6;
        else
            this.len = 7;
    }
    byteLength() {
        return 1 + this.len;
    }
}
export class NintAstNode {
    val;
    len;
    constructor(val) {
        this.val = val;
        const uint = -val;
        if (!uint)
            this.len = 0;
        else if (uint <= 0xff)
            this.len = 1;
        else if (uint <= 0xffff)
            this.len = 2;
        else if (uint <= 0xffffff)
            this.len = 3;
        else if (uint <= 0xffffffff)
            this.len = 4;
        else if (uint <= 0xffffffffff)
            this.len = 5;
        else if (uint <= 0xffffffffffff)
            this.len = 6;
        else
            this.len = 7;
    }
    byteLength() {
        return 1 + this.len;
    }
}
export class FloatAstNode {
    val;
    len = 8;
    constructor(val) {
        this.val = val;
    }
    byteLength() {
        return 1 + this.len;
    }
}
const vUintLen = (num) => {
    if (num <= 0b1111111)
        return 1;
    else if (num <= 0b1111111_1111111)
        return 2;
    else if (num <= 0b1111111_1111111_1111111)
        return 3;
    else if (num <= 0b1111111_1111111_1111111_1111111)
        return 4;
    else if (num <= 0b1111111_1111111_1111111_1111111_1111111)
        return 5;
    else
        return 6;
};
export class StrAstNode {
    val;
    len;
    constructor(val) {
        this.val = val;
        this.len = utf8Count(val);
    }
    byteLength() {
        return this.len < 14 ? 1 + this.len : 1 + vUintLen(this.len) + this.len;
    }
}
export class BinAstNode {
    val;
    len;
    constructor(val) {
        this.val = val;
        this.len = val.length;
    }
    byteLength() {
        return this.len < 14 ? 1 + this.len : 1 + vUintLen(this.len) + this.len;
    }
}
export class ArrAstNode {
    val;
    len;
    constructor(val) {
        this.val = val;
        if (val === null) {
            this.len = 1;
        }
        else {
            if (!val.length)
                this.len = 0;
            else {
                let elementLength = 0;
                for (let i = 0; i < val.length; i++)
                    elementLength += val[i].byteLength();
                this.len = elementLength;
            }
        }
    }
    byteLength() {
        return this.len < 14 ? 1 + this.len : 1 + vUintLen(this.len) + this.len;
    }
}
export class ObjAstNode {
    val;
    len;
    constructor(val) {
        this.val = val;
        if (val === null) {
            this.len = 1;
        }
        else {
            if (!val.size)
                this.len = 0;
            else {
                let len = 0;
                val.forEach((node, symbolId) => {
                    len += vUintLen(symbolId) + node.byteLength();
                });
                this.len = len;
            }
        }
    }
    byteLength() {
        return this.len < 14 ? 1 + this.len : 1 + vUintLen(this.len) + this.len;
    }
}
export class AnnotationAstNode {
    val;
    annotations;
    len;
    annotationLen;
    constructor(val, annotations) {
        this.val = val;
        this.annotations = annotations;
        let len = 0;
        for (let i = 0; i < annotations.length; i++)
            len += vUintLen(annotations[i]);
        this.annotationLen = len;
        len += vUintLen(len);
        len += val.byteLength();
        this.len = len;
    }
    byteLength() {
        return this.len < 14 ? 1 + this.len : 1 + vUintLen(this.len) + this.len;
    }
}
const isSafeInteger = Number.isSafeInteger;
export const toAst = (val, symbols) => {
    if (val === null)
        return new NullAstNode();
    if (val instanceof Array)
        return new ArrAstNode(val.map((el) => toAst(el, symbols)));
    if (val instanceof Uint8Array)
        return new BinAstNode(val);
    switch (typeof val) {
        case 'boolean':
            return new BoolAstNode(val);
        case 'number': {
            if (isSafeInteger(val))
                return val >= 0 ? new UintAstNode(val) : new NintAstNode(val);
            else
                return new FloatAstNode(val);
        }
        case 'string':
            return new StrAstNode(val);
        case 'object': {
            const struct = new Map();
            for (const key in val) {
                const symbolId = symbols.add(key);
                struct.set(symbolId, toAst(val[key], symbols));
            }
            return new ObjAstNode(struct);
        }
    }
    throw new Error('UNKNOWN_TYPE');
};
