import { wordWrap } from '../../util/strings/wordWrap';
import { TAB, isSimpleType, normalizeKey } from './util';
const formatComment = (comment, __) => {
    if (!comment)
        return '';
    const lines = wordWrap(comment, { width: 80 - 3 - __.length });
    return __ + '/**\n' + __ + ' * ' + lines.join('\n' + __ + ' * ') + '\n' + __ + ' */\n';
};
export const toText = (node, __ = '') => {
    if (Array.isArray(node))
        return node.map((s) => toText(s, __)).join('\n');
    const ____ = __ + TAB;
    switch (node.node) {
        case 'ModuleDeclaration': {
            let out = '';
            out += `${__}namespace ${node.name} {\n`;
            out += toText(node.statements, ____);
            out += `${__}}\n`;
            return out;
        }
        case 'InterfaceDeclaration': {
            const { name, members, comment } = node;
            let out = '';
            out += formatComment(comment, __);
            out += `${__}interface ${name} {\n`;
            out += toText(members, ____);
            out += `\n${__}}\n`;
            return out;
        }
        case 'PropertySignature': {
            const name = normalizeKey(node.name);
            let out = '';
            out += formatComment(node.comment, __);
            return out + `${__}${name}${node.optional ? '?' : ''}: ${toText(node.type, __)};`;
        }
        case 'IndexSignature': {
            return `${__}[key: string]: ${toText(node.type, __)};`;
        }
        case 'TypeAliasDeclaration': {
            let out = '';
            out += formatComment(node.comment, __);
            out += `${__}type ${node.name} = ${toText(node.type)};\n`;
            return out;
        }
        case 'ArrayType': {
            const simple = isSimpleType(node.elementType);
            const inner = toText(node.elementType, __);
            return simple ? `${inner}[]` : `Array<${inner}>`;
        }
        case 'TupleType': {
            const hasObject = node.elements.some((e) => e.node === 'TypeLiteral');
            if (hasObject) {
                return `[\n${____}${node.elements.map((e) => toText(e, ____)).join(',\n' + ____)}\n${__}]`;
            }
            else
                return `[${node.elements.map((e) => toText(e, __)).join(', ')}]`;
        }
        case 'GenericTypeAnnotation': {
            return node.id.name;
        }
        case 'StringKeyword': {
            return 'string';
        }
        case 'NumberKeyword': {
            return 'number';
        }
        case 'BooleanKeyword': {
            return 'boolean';
        }
        case 'NullKeyword': {
            return 'null';
        }
        case 'AnyKeyword': {
            return 'any';
        }
        case 'UnknownKeyword': {
            return 'unknown';
        }
        case 'TypeLiteral': {
            return `{\n${toText(node.members, ____)}\n${__}}`;
        }
        case 'StringLiteral': {
            return JSON.stringify(node.text);
        }
        case 'NumericLiteral': {
            return node.text;
        }
        case 'TrueKeyword': {
            return 'true';
        }
        case 'FalseKeyword': {
            return 'false';
        }
        case 'UnionType': {
            return node.types.map((t) => toText(t, ____)).join(' | ');
        }
        case 'TypeReference': {
            return node.typeName;
        }
    }
    return '';
};
