import { Subject, ReplaySubject, BehaviorSubject, from } from 'rxjs';
import { toUint8Array } from '../../../util/buffers/toUint8Array';
import { delay, filter, map, skip, switchMap, take, takeUntil, tap } from 'rxjs/operators';
export class WebSocketChannel {
    ws;
    state$ = new BehaviorSubject(0);
    open$ = new ReplaySubject(1);
    close$ = new ReplaySubject(1);
    error$ = new Subject();
    message$ = new Subject();
    constructor({ newSocket }) {
        try {
            const ws = (this.ws = newSocket());
            ws.binaryType = 'arraybuffer';
            ws.onopen = () => {
                this.state$.next(1);
                this.open$.next(this);
                this.open$.complete();
            };
            ws.onclose = (event) => {
                this.state$.next(2);
                this.close$.next([this, event]);
                this.close$.complete();
                this.message$.complete();
            };
            ws.onerror = (event) => {
                const errorEvent = event;
                const error = errorEvent.error instanceof Error ? errorEvent.error : new Error(String(errorEvent.message) || 'ERROR');
                this.error$.next(error);
            };
            ws.onmessage = (event) => {
                const data = event.data;
                const message = (typeof data === 'string' ? data : toUint8Array(data));
                this.message$.next(message);
            };
        }
        catch (error) {
            this.state$.next(2);
            this.error$.next(error);
            this.close$.next([this, { code: 0, wasClean: true, reason: 'CONSTRUCTOR' }]);
            this.close$.complete();
        }
    }
    buffer() {
        if (!this.ws)
            return 0;
        return this.ws.bufferedAmount;
    }
    close(code, reason) {
        if (!this.ws)
            return;
        this.ws.close(code, reason);
    }
    isOpen() {
        return this.state$.getValue() === 1;
    }
    send(data) {
        if (!this.ws)
            return -1;
        const buffered = this.ws.bufferedAmount;
        this.ws.send(data);
        return this.ws.bufferedAmount - buffered;
    }
    send$(data) {
        return this.open$.pipe(map(() => {
            if (!this.isOpen())
                throw new Error('CLOSED');
            return this.send(data);
        }));
    }
}
export class PersistentChannel {
    params;
    active$ = new BehaviorSubject(false);
    channel$ = new BehaviorSubject(undefined);
    open$ = new BehaviorSubject(false);
    message$ = this.channel$.pipe(filter((channel) => !!channel), switchMap((channel) => channel.message$));
    retries = 0;
    constructor(params) {
        this.params = params;
        const start$ = new Subject();
        const stop$ = new Subject();
        this.active$
            .pipe(skip(1), filter((active) => active))
            .subscribe(() => {
            start$.next(undefined);
        });
        this.active$
            .pipe(skip(1), filter((active) => !active))
            .subscribe(() => {
            stop$.next(undefined);
        });
        start$.subscribe(() => this.channel$.next(params.newChannel()));
        start$
            .pipe(switchMap(() => this.channel$), filter((channel) => !!channel), takeUntil(stop$), switchMap((channel) => channel.close$), takeUntil(stop$), switchMap(() => from((async () => {
            const timeout = this.reconnectDelay();
            this.retries++;
            await new Promise((resolve) => setTimeout(resolve, timeout));
        })())), takeUntil(stop$), tap(() => this.channel$.next(params.newChannel())), delay(params.minUptime || 5_000), takeUntil(stop$), tap(() => {
            const isOpen = this.channel$.getValue()?.isOpen();
            if (isOpen) {
                this.retries = 0;
            }
        }))
            .subscribe();
        start$
            .pipe(switchMap(() => this.channel$), filter((channel) => !!channel), switchMap((channel) => channel.state$), map((state) => state === 1))
            .subscribe((open) => {
            if (open !== this.open$.getValue())
                this.open$.next(open);
        });
        stop$.subscribe(() => {
            this.retries = 0;
        });
    }
    start() {
        if (this.active$.getValue())
            return;
        this.active$.next(true);
    }
    stop() {
        if (!this.active$.getValue())
            return;
        this.active$.next(false);
        const channel = this.channel$.getValue();
        if (channel) {
            channel.close();
            this.channel$.next(undefined);
        }
        this.open$.next(false);
    }
    reconnectDelay() {
        if (this.retries <= 0)
            return 0;
        const minReconnectionDelay = this.params.minReconnectionDelay || Math.round(1_000 + Math.random() * 1_000);
        const maxReconnectionDelay = this.params.maxReconnectionDelay || 10_000;
        const reconnectionDelayGrowFactor = this.params.reconnectionDelayGrowFactor || 1.3;
        const delay = Math.min(maxReconnectionDelay, minReconnectionDelay * reconnectionDelayGrowFactor ** (this.retries - 1));
        return delay;
    }
    send$(data) {
        return this.channel$.pipe(filter((channel) => !!channel), switchMap((channel) => channel.open$), filter((channel) => channel.isOpen()), take(1), map((channel) => {
            const canSend = this.active$.getValue() && this.open$.getValue();
            return canSend ? channel.send(data) : -1;
        }));
    }
}
