import { validateId, validateMethod } from '../rpc/validation';
import { CborEncoder } from '../../../json-pack/cbor/CborEncoder';
import { MsgPackEncoder } from '../../../json-pack/msgpack';
import { JsonEncoder } from '../../../json-pack/json/JsonEncoder';
const encodeHeader = (writer, typeU16, id, payloadSize, start) => {
    if (payloadSize <= 0b1111_11111111) {
        const w1 = typeU16 | payloadSize;
        const w2 = id;
        writer.view.setUint32(start, (w1 << 16) | w2);
    }
    else if (payloadSize <= 0b1111_11111111_1111111) {
        writer.u8(id & 0xff);
        const w1 = typeU16 | 0b000_1_0000_00000000 | (payloadSize >> 7);
        const w2 = ((payloadSize & 0b0111_1111) << 8) | (id >> 8);
        writer.view.setUint32(start, (w1 << 16) | w2);
    }
    else {
        writer.u16(id);
        const w1 = typeU16 | 0b000_1_0000_00000000 | (payloadSize >> 15);
        const w2 = 0b1000_0000_00000000 | (payloadSize & 0b0111_1111_11111111);
        writer.view.setUint32(start, (w1 << 16) | w2);
    }
};
const encodeBinaryWithNameAndPayload = (codec, typeU16, id, name, value) => {
    const writer = codec.encoder.writer;
    const nameLength = name.length;
    writer.ensureCapacity(5 + nameLength);
    writer.uint8[writer.x + 4] = nameLength;
    writer.x += 5;
    writer.ascii(name);
    const x0 = writer.x0;
    const x = writer.x;
    if (value)
        value.encode(codec);
    const shift = writer.x0 - x0;
    const payloadStart = x + shift;
    const start = payloadStart - 5 - nameLength;
    const payloadSize = writer.x - payloadStart;
    encodeHeader(writer, typeU16, id, payloadSize, start);
};
const encodeBinaryWithPayload = (codec, typeU16, id, value) => {
    const writer = codec.encoder.writer;
    writer.move(4);
    const x0 = writer.x0;
    const x = writer.x;
    if (value)
        value.encode(codec);
    const shift = writer.x0 - x0;
    const payloadStart = x + shift;
    const start = payloadStart - 4;
    const payloadSize = writer.x - payloadStart;
    encodeHeader(writer, typeU16, id, payloadSize, start);
};
const encodeCompactWithNameAndPayload = (codec, type, msg) => {
    const encoder = codec.encoder;
    if (encoder instanceof CborEncoder || encoder instanceof MsgPackEncoder) {
        const value = msg.value;
        const hasValue = value !== undefined;
        encoder.writeArrHdr(hasValue ? 4 : 3);
        encoder.writeUInteger(type);
        encoder.writeUInteger(msg.id);
        encoder.writeAsciiStr(msg.method);
        if (hasValue) {
            if (value.type)
                value.type.encoder(codec.format)(value.data, encoder);
            else
                encoder.writeAny(value.data);
        }
    }
    else if (encoder instanceof JsonEncoder) {
        const value = msg.value;
        encoder.writeStartArr();
        encoder.writeNumber(type);
        encoder.writeArrSeparator();
        encoder.writeNumber(msg.id);
        encoder.writeArrSeparator();
        encoder.writeAsciiStr(msg.method);
        const hasValue = value !== undefined;
        if (hasValue) {
            encoder.writeArrSeparator();
            if (value.type)
                value.type.encoder(codec.format)(value.data, encoder);
            else
                encoder.writeAny(value.data);
        }
        encoder.writeEndArr();
    }
    else
        encoder.writeArr(msg.toCompact());
};
const encodeCompactWithPayload = (codec, type, msg) => {
    const encoder = codec.encoder;
    if (encoder instanceof CborEncoder || encoder instanceof MsgPackEncoder) {
        const value = msg.value;
        const hasValue = value !== undefined;
        encoder.writeArrHdr(hasValue ? 3 : 2);
        encoder.writeUInteger(type);
        encoder.writeUInteger(msg.id);
        if (hasValue) {
            if (value.type) {
                value.type.encoder(codec.format)(value.data, encoder);
            }
            else
                encoder.writeAny(value.data);
        }
    }
    else if (encoder instanceof JsonEncoder) {
        const value = msg.value;
        encoder.writeStartArr();
        encoder.writeNumber(type);
        encoder.writeArrSeparator();
        encoder.writeNumber(msg.id);
        const hasValue = value !== undefined;
        if (hasValue) {
            encoder.writeArrSeparator();
            if (value.type)
                value.type.encoder(codec.format)(value.data, encoder);
            else
                encoder.writeAny(value.data);
        }
        encoder.writeEndArr();
    }
    else
        encoder.writeArr(msg.toCompact());
};
export class NotificationMessage {
    method;
    value;
    constructor(method, value) {
        this.method = method;
        this.value = value;
    }
    validate() {
        validateMethod(this.method);
    }
    toCompact() {
        return this.value === undefined
            ? [8, this.method]
            : [8, this.method, this.value.data];
    }
    encodeCompact(codec) {
        const encoder = codec.encoder;
        if (encoder instanceof CborEncoder || encoder instanceof MsgPackEncoder) {
            const value = this.value;
            const hasValue = value !== undefined;
            encoder.writeArrHdr(hasValue ? 3 : 2);
            encoder.writeUInteger(8);
            encoder.writeAsciiStr(this.method);
            if (hasValue) {
                if (value.type)
                    value.type.encoder(codec.format)(value.data, encoder);
                else
                    encoder.writeAny(value.data);
            }
        }
        else if (encoder instanceof JsonEncoder) {
            const value = this.value;
            encoder.writeStartArr();
            encoder.writeNumber(8);
            encoder.writeArrSeparator();
            encoder.writeAsciiStr(this.method);
            const hasValue = value !== undefined;
            if (hasValue) {
                encoder.writeArrSeparator();
                if (value.type)
                    value.type.encoder(codec.format)(value.data, encoder);
                else
                    encoder.writeAny(value.data);
            }
            encoder.writeEndArr();
        }
        else
            encoder.writeArr(this.toCompact());
    }
    encodeBinary(codec) {
        const writer = codec.encoder.writer;
        const name = this.method;
        const nameLength = name.length;
        writer.move(4);
        writer.ascii(name);
        const x0 = writer.x0;
        const x = writer.x;
        this.value.encode(codec);
        const shift = writer.x0 - x0;
        const payloadStart = x + shift;
        const start = payloadStart - 4 - nameLength;
        const payloadSize = writer.x - payloadStart;
        writer.view.setUint32(start, (payloadSize << 8) + nameLength);
    }
}
export class RequestDataMessage {
    id;
    method;
    value;
    constructor(id, method, value) {
        this.id = id;
        this.method = method;
        this.value = value;
    }
    validate() {
        validateId(this.id);
        if (this.method)
            validateMethod(this.method);
    }
    toCompact() {
        return this.value === undefined
            ? [0, this.id, this.method]
            : [0, this.id, this.method, this.value.data];
    }
    encodeCompact(codec) {
        encodeCompactWithNameAndPayload(codec, 0, this);
    }
    encodeBinary(codec) {
        encodeBinaryWithNameAndPayload(codec, 1 << 13, this.id, this.method, this.value);
    }
}
export class RequestCompleteMessage {
    id;
    method;
    value;
    constructor(id, method, value) {
        this.id = id;
        this.method = method;
        this.value = value;
    }
    validate() {
        validateId(this.id);
        if (this.method)
            validateMethod(this.method);
    }
    toCompact() {
        return this.value === undefined
            ? [1, this.id, this.method]
            : [1, this.id, this.method, this.value.data];
    }
    encodeCompact(codec) {
        encodeCompactWithNameAndPayload(codec, 1, this);
    }
    encodeBinary(codec) {
        encodeBinaryWithNameAndPayload(codec, 2 << 13, this.id, this.method, this.value);
    }
}
export class RequestErrorMessage {
    id;
    method;
    value;
    constructor(id, method, value) {
        this.id = id;
        this.method = method;
        this.value = value;
    }
    validate() {
        validateId(this.id);
        if (this.method)
            validateMethod(this.method);
    }
    toCompact() {
        return [2, this.id, this.method, this.value.data];
    }
    encodeCompact(codec) {
        encodeCompactWithNameAndPayload(codec, 2, this);
    }
    encodeBinary(codec) {
        encodeBinaryWithNameAndPayload(codec, 3 << 13, this.id, this.method, this.value);
    }
}
export class RequestUnsubscribeMessage {
    id;
    constructor(id) {
        this.id = id;
    }
    validate() {
        validateId(this.id);
    }
    toCompact() {
        return [3, this.id];
    }
    encodeCompact(codec) {
        codec.encoder.writeArr(this.toCompact());
    }
    encodeBinary(codec) {
        codec.encoder.writer.u32(0b11100000_00000000_00000000_00000000 | this.id);
    }
}
export class ResponseCompleteMessage {
    id;
    value;
    constructor(id, value) {
        this.id = id;
        this.value = value;
    }
    validate() {
        validateId(this.id);
    }
    toCompact() {
        return this.value === undefined
            ? [5, this.id]
            : [5, this.id, this.value.data];
    }
    encodeCompact(codec) {
        encodeCompactWithPayload(codec, 5, this);
    }
    encodeBinary(codec) {
        encodeBinaryWithPayload(codec, 5 << 13, this.id, this.value);
    }
}
export class ResponseDataMessage {
    id;
    value;
    constructor(id, value) {
        this.id = id;
        this.value = value;
    }
    validate() {
        validateId(this.id);
    }
    toCompact() {
        return [4, this.id, this.value.data];
    }
    encodeCompact(codec) {
        encodeCompactWithPayload(codec, 4, this);
    }
    encodeBinary(codec) {
        encodeBinaryWithPayload(codec, 4 << 13, this.id, this.value);
    }
}
export class ResponseErrorMessage {
    id;
    value;
    constructor(id, value) {
        this.id = id;
        this.value = value;
    }
    validate() {
        validateId(this.id);
    }
    toCompact() {
        return [6, this.id, this.value.data];
    }
    encodeCompact(codec) {
        encodeCompactWithPayload(codec, 6, this);
    }
    encodeBinary(codec) {
        encodeBinaryWithPayload(codec, 6 << 13, this.id, this.value);
    }
}
export class ResponseUnsubscribeMessage {
    id;
    constructor(id) {
        this.id = id;
    }
    validate() {
        validateId(this.id);
    }
    toCompact() {
        return [7, this.id];
    }
    encodeCompact(codec) {
        codec.encoder.writeArr(this.toCompact());
    }
    encodeBinary(codec) {
        codec.encoder.writer.u32(0b11100000_00000001_00000000_00000000 | this.id);
    }
}
