import { firstValueFrom, from, Subject } from 'rxjs';
import { catchError, finalize, first, map, mergeWith, share, switchMap, take, tap } from 'rxjs/operators';
import { BufferSubject } from '../../../../util/rx/BufferSubject';
import { RpcError, RpcErrorCodes } from './error';
import { Value } from '../../messages/Value';
import { StaticRpcMethod } from '../methods/StaticRpcMethod';
const INVALID_REQUEST_ERROR_VALUE = RpcError.value(RpcError.invalidRequest());
const defaultWrapInternalError = (error) => RpcError.valueFrom(error);
export class RpcCaller {
    getMethod;
    preCallBufferSize;
    wrapInternalError;
    constructor({ getMethod, preCallBufferSize = 10, wrapInternalError = defaultWrapInternalError, }) {
        this.getMethod = getMethod;
        this.preCallBufferSize = preCallBufferSize;
        this.wrapInternalError = wrapInternalError;
    }
    exists(name) {
        return !!this.getMethod(name);
    }
    getMethodStrict(name) {
        const method = this.getMethod(name);
        if (!method)
            throw RpcError.valueFromCode(RpcErrorCodes.METHOD_NOT_FOUND);
        return method;
    }
    info(name) {
        return this.getMethodStrict(name);
    }
    validate(method, request) {
        const validate = method.validate;
        if (!validate)
            return;
        try {
            const errors = validate(request);
            if (errors)
                throw errors;
        }
        catch (error) {
            throw this.wrapValidationError(error);
        }
    }
    wrapValidationError(error) {
        return RpcError.valueFrom(error, INVALID_REQUEST_ERROR_VALUE);
    }
    async call(name, request, ctx) {
        try {
            const method = this.getMethodStrict(name);
            this.validate(method, request);
            const preCall = method.onPreCall;
            if (preCall)
                await preCall(ctx, request);
            const data = await method.call(request, ctx);
            return new Value(data, method.res);
        }
        catch (error) {
            throw this.wrapInternalError(error);
        }
    }
    async notification(name, request, ctx) {
        const method = this.getMethodStrict(name);
        if (!(method instanceof StaticRpcMethod))
            return;
        if (!method.acceptsNotifications)
            return;
        this.validate(method, request);
        try {
            if (method.onPreCall)
                await method.onPreCall(ctx, request);
            await method.call(request, ctx);
        }
        catch (error) {
            throw this.wrapInternalError(error);
        }
    }
    createCall(name, ctx) {
        const req$ = new Subject();
        const reqUnsubscribe$ = new Subject();
        try {
            const method = this.getMethodStrict(name);
            if (!method.isStreaming) {
                const response$ = from((async () => {
                    const request = await firstValueFrom(req$.pipe(first()));
                    return await this.call(name, request, ctx);
                })());
                const res$ = new Subject();
                response$.subscribe(res$);
                const $resWithErrorsFormatted = res$.pipe(catchError((error) => {
                    throw this.wrapInternalError(error);
                }));
                return { req$, reqUnsubscribe$, res$: $resWithErrorsFormatted };
            }
            const methodStreaming = method;
            const requestValidated$ = req$.pipe(tap((request) => {
                this.validate(methodStreaming, request);
            }));
            const bufferSize = methodStreaming.preCallBufferSize || this.preCallBufferSize;
            const requestBuffered$ = new BufferSubject(bufferSize);
            const error$ = new Subject();
            requestBuffered$.subscribe({
                error: (error) => {
                    error$.error(error);
                },
            });
            requestValidated$.subscribe(requestBuffered$);
            const methodResponseType = method.res;
            const result$ = requestBuffered$.pipe(take(1), switchMap((request) => {
                return methodStreaming.onPreCall ? from(methodStreaming.onPreCall(ctx, request)) : from([0]);
            }), switchMap(() => {
                Promise.resolve().then(() => {
                    requestBuffered$.flush();
                });
                return method.call$(requestBuffered$, ctx).pipe(map((response) => new Value(response, methodResponseType)), finalize(() => {
                    error$.complete();
                }));
            }), share(), mergeWith(error$));
            const $resWithErrorsFormatted = result$.pipe(finalize(() => {
                error$.complete();
            }), catchError((error) => {
                throw RpcError.valueFrom(error);
            }));
            return { req$, reqUnsubscribe$, res$: $resWithErrorsFormatted };
        }
        catch (error) {
            const errorFormatted = RpcError.valueFrom(error);
            req$.error(errorFormatted);
            const res$ = new Subject();
            res$.error(errorFormatted);
            return { req$, reqUnsubscribe$, res$ };
        }
    }
    call$(name, request$, ctx) {
        const call = this.createCall(name, ctx);
        request$.subscribe(call.req$);
        return call.res$;
    }
}
