import { RpcError, RpcErrorCodes } from './error';
import { RpcCaller } from './RpcCaller';
import { FunctionStreamingType, FunctionType } from '../../../../json-type/type/classes';
import { StaticRpcMethod } from '../methods/StaticRpcMethod';
import { StreamingRpcMethod } from '../methods/StreamingRpcMethod';
export class TypedApiCaller extends RpcCaller {
    system;
    methods = new Map();
    constructor({ system, ...rest }) {
        super({
            ...rest,
            getMethod: (name) => this.get(name),
        });
        this.system = system;
    }
    req = null;
    res = null;
    implement(id, definition_) {
        const definition = definition_;
        if (this.methods.has(id))
            throw new Error(`Method [id = ${id}] is already implemented.`);
        const alias = this.system.resolve(id);
        const type = alias.type;
        if (!(type instanceof FunctionType || type instanceof FunctionStreamingType))
            throw new Error(`Type [alias = ${alias.id}] is not a function.`);
        const validator = type.validator('boolean');
        const customValidator = definition.validate;
        const validate = customValidator
            ? (req) => {
                const error = validator(req);
                if (error)
                    throw RpcError.valueFromCode(RpcErrorCodes.BAD_REQUEST);
                customValidator(req);
            }
            : (req) => {
                const error = validator(req);
                if (error)
                    throw RpcError.valueFromCode(RpcErrorCodes.BAD_REQUEST);
            };
        const isStaticMethodAlias = alias.type instanceof FunctionType;
        const isStreamingMethodAlias = alias.type instanceof FunctionStreamingType;
        const method = isStaticMethodAlias
            ? new StaticRpcMethod({
                ...definition,
                req: type.req,
                res: type.res,
                validate,
            })
            : isStreamingMethodAlias
                ? new StreamingRpcMethod({
                    ...definition,
                    req: type.req,
                    res: type.res,
                    validate,
                })
                : null;
        if (!method)
            throw new Error(`Type [alias = ${alias.id}] is not a function.`);
        this.methods.set(id, method);
    }
    get(id) {
        const method = this.methods.get(id);
        if (!method)
            throw RpcError.valueFromCode(RpcErrorCodes.METHOD_NOT_FOUND);
        return method;
    }
}
