import { NullObject } from '../../util/NullObject';
import { copy } from '../../util/buffers/copy';
import { listToUint8 } from '../../util/buffers/concat';
const X_AUTH_PARAM = 'X-Authorization=';
const X_AUTH_PARAM_LENGTH = X_AUTH_PARAM.length;
const CODECS_REGEX = /rpc.(\w{0,32})\.(\w{0,32})\.(\w{0,32})(?:\-(\w{0,32}))?/;
export class ConnectionContext {
    ip;
    token;
    params;
    meta;
    reqCodec;
    resCodec;
    msgCodec;
    res;
    static fromReqRes(req, res, params, app) {
        const ip = req.getHeader('x-forwarded-for') ||
            req.getHeader('x-real-ip') ||
            Buffer.from(res.getRemoteAddressAsText()).toString();
        let token = req.getHeader('authorization') || '';
        if (!token) {
            const query = req.getQuery();
            const params = new URLSearchParams(query);
            token = params.get('access_token') || '';
            if (!token)
                token = params.get('token') || '';
        }
        let secWebSocketProtocol = '';
        if (!token) {
            secWebSocketProtocol = String(req.getHeader('sec-websocket-protocol')) || '';
            if (secWebSocketProtocol) {
                const protocols = secWebSocketProtocol.split(',');
                const length = protocols.length;
                for (let i = 0; i < length; i++) {
                    let protocol = protocols[i].trim();
                    if (protocol.startsWith(X_AUTH_PARAM)) {
                        protocol = protocol.slice(X_AUTH_PARAM_LENGTH);
                        if (protocol) {
                            token = Buffer.from(protocol, 'base64').toString();
                            break;
                        }
                    }
                }
            }
        }
        const codecs = app.codecs;
        const valueCodecs = codecs.value;
        const ctx = new ConnectionContext(ip, token, params, new NullObject(), valueCodecs.json, valueCodecs.json, codecs.messages.compact, res);
        const contentType = req.getHeader('content-type');
        if (contentType)
            ctx.setCodecs(contentType, codecs);
        else if (secWebSocketProtocol)
            ctx.setCodecs(secWebSocketProtocol, codecs);
        return ctx;
    }
    constructor(ip, token, params, meta, reqCodec, resCodec, msgCodec, res = undefined) {
        this.ip = ip;
        this.token = token;
        this.params = params;
        this.meta = meta;
        this.reqCodec = reqCodec;
        this.resCodec = resCodec;
        this.msgCodec = msgCodec;
        this.res = res;
    }
    setCodecs(specifier, codecs) {
        const match = CODECS_REGEX.exec(specifier);
        if (!match)
            return;
        const [, protocol, messageFormat, request, response] = match;
        switch (protocol) {
            case 'rx': {
                switch (messageFormat) {
                    case 'compact': {
                        this.msgCodec = codecs.messages.compact;
                        break;
                    }
                    case 'binary': {
                        this.msgCodec = codecs.messages.binary;
                        break;
                    }
                }
                break;
            }
            case 'json2': {
                this.msgCodec = codecs.messages.jsonRpc2;
                break;
            }
        }
        switch (request) {
            case 'cbor': {
                this.resCodec = this.reqCodec = codecs.value.cbor;
                break;
            }
            case 'json': {
                this.resCodec = this.reqCodec = codecs.value.json;
                break;
            }
            case 'msgpack': {
                this.resCodec = this.reqCodec = codecs.value.msgpack;
                break;
            }
        }
        switch (response) {
            case 'cbor': {
                this.resCodec = codecs.value.cbor;
                break;
            }
            case 'json': {
                this.resCodec = codecs.value.json;
                break;
            }
            case 'msgpack': {
                this.resCodec = codecs.value.msgpack;
                break;
            }
        }
    }
    requestBodyParts(max) {
        const res = this.res;
        return new Promise((resolve) => {
            const list = [];
            if (!res)
                return resolve(list);
            let running = 0;
            res.onData((ab, isLast) => {
                running += ab.byteLength;
                if (running > max)
                    res.end('too large');
                if (isLast)
                    list.push(new Uint8Array(ab)), resolve(list);
                else
                    list.push(copy(new Uint8Array(ab)));
            });
        });
    }
    async requestBody(max) {
        const parts = await this.requestBodyParts(max);
        return listToUint8(parts);
    }
    async requestBodyJson(max) {
        const parts = await this.requestBodyParts(max);
        const bodyUint8 = listToUint8(parts);
        return this.reqCodec.decoder.read(bodyUint8);
    }
    sendResponse(response) {
        const res = this.res;
        if (!res)
            return;
        if (res.aborted)
            return;
        res.end(response);
    }
}
