"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Patch = void 0;
const tslib_1 = require("tslib");
const operations = tslib_1.__importStar(require("./operations"));
const clock_1 = require("./clock");
const binary_1 = require("./codec/binary");
class Patch {
    constructor() {
        this.ops = [];
        this.meta = undefined;
    }
    static fromBinary(data) {
        return (0, binary_1.decode)(data);
    }
    getId() {
        const op = this.ops[0];
        if (!op)
            return undefined;
        return op.id;
    }
    span() {
        let span = 0;
        for (const op of this.ops)
            span += op.span();
        return span;
    }
    nextTime() {
        const ops = this.ops;
        const length = ops.length;
        if (!length)
            return 0;
        const lastOp = ops[length - 1];
        return lastOp.id.time + lastOp.span();
    }
    rewriteTime(ts) {
        const patch = new Patch();
        const ops = this.ops;
        const length = ops.length;
        const patchOps = patch.ops;
        for (let i = 0; i < length; i++) {
            const op = ops[i];
            if (op instanceof operations.DelOp)
                patchOps.push(new operations.DelOp(ts(op.id), ts(op.obj), op.what));
            else if (op instanceof operations.NewConOp)
                patchOps.push(new operations.NewConOp(ts(op.id), op.val));
            else if (op instanceof operations.NewVecOp)
                patchOps.push(new operations.NewVecOp(ts(op.id)));
            else if (op instanceof operations.NewValOp)
                patchOps.push(new operations.NewValOp(ts(op.id)));
            else if (op instanceof operations.NewObjOp)
                patchOps.push(new operations.NewObjOp(ts(op.id)));
            else if (op instanceof operations.NewStrOp)
                patchOps.push(new operations.NewStrOp(ts(op.id)));
            else if (op instanceof operations.NewBinOp)
                patchOps.push(new operations.NewBinOp(ts(op.id)));
            else if (op instanceof operations.NewArrOp)
                patchOps.push(new operations.NewArrOp(ts(op.id)));
            else if (op instanceof operations.InsArrOp)
                patchOps.push(new operations.InsArrOp(ts(op.id), ts(op.obj), ts(op.ref), op.data.map(ts)));
            else if (op instanceof operations.InsStrOp)
                patchOps.push(new operations.InsStrOp(ts(op.id), ts(op.obj), ts(op.ref), op.data));
            else if (op instanceof operations.InsBinOp)
                patchOps.push(new operations.InsBinOp(ts(op.id), ts(op.obj), ts(op.ref), op.data));
            else if (op instanceof operations.InsValOp)
                patchOps.push(new operations.InsValOp(ts(op.id), ts(op.obj), ts(op.val)));
            else if (op instanceof operations.InsObjOp)
                patchOps.push(new operations.InsObjOp(ts(op.id), ts(op.obj), op.data.map(([key, value]) => [key, ts(value)])));
            else if (op instanceof operations.NopOp)
                patchOps.push(new operations.NopOp(ts(op.id), op.len));
        }
        return patch;
    }
    rebase(serverTime, transformHorizon) {
        const id = this.getId();
        if (!id)
            throw new Error('EMPTY_PATCH');
        const patchStartTime = id.time;
        if (patchStartTime === serverTime)
            return this;
        const delta = serverTime - patchStartTime;
        return this.rewriteTime((id) => {
            const sessionId = id.sid;
            const isServerTimestamp = sessionId === 1;
            if (!isServerTimestamp)
                return id;
            const time = id.time;
            if (time < transformHorizon)
                return id;
            return (0, clock_1.ts)(1, time + delta);
        });
    }
    clone() {
        return this.rewriteTime((id) => id);
    }
    toBinary() {
        return (0, binary_1.encode)(this);
    }
    toString(tab = '') {
        const id = this.getId();
        let out = `${this.constructor.name} ${id ? (0, clock_1.toDisplayString)(id) : '(nil)'}!${this.span()}`;
        for (let i = 0; i < this.ops.length; i++) {
            const isLast = i === this.ops.length - 1;
            out += `\n${tab}${isLast ? '└─' : '├─'} ${this.ops[i].toString(tab + (isLast ? '  ' : '│ '))}`;
        }
        return out;
    }
}
exports.Patch = Patch;
