"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonPatch = void 0;
const deepEqual_1 = require("../../json-equal/deepEqual");
const json_pointer_1 = require("../../json-pointer");
const nodes_1 = require("../nodes");
const util_1 = require("../../json-pointer/util");
const clock_1 = require("../../json-crdt-patch/clock");
class JsonPatch {
    constructor(model) {
        this.model = model;
    }
    apply(ops) {
        const length = ops.length;
        for (let i = 0; i < length; i++)
            this.applyOp(ops[i]);
        return this;
    }
    applyOp(op) {
        switch (op.op) {
            case 'add':
                this.applyOpAdd(op);
                break;
            case 'remove':
                this.applyRemove(op);
                break;
            case 'replace':
                this.applyReplace(op);
                break;
            case 'move':
                this.applyMove(op);
                break;
            case 'copy':
                this.applyCopy(op);
                break;
            case 'test':
                this.applyTest(op);
                break;
            case 'str_ins':
                this.applyStrIns(op);
                break;
            case 'str_del':
                this.applyStrDel(op);
                break;
            default:
                throw new Error('UNKNOWN_OP');
        }
        this.model.api.apply();
        return this;
    }
    builder() {
        return this.model.api.builder;
    }
    applyOpAdd(op) {
        const builder = this.builder();
        const steps = (0, util_1.toPath)(op.path);
        if (!steps.length)
            this.setRoot(op.value);
        else {
            const objSteps = steps.slice(0, steps.length - 1);
            const node = this.model.api.find(objSteps);
            const key = steps[steps.length - 1];
            if (node instanceof nodes_1.ObjNode) {
                builder.insObj(node.id, [[String(key), builder.json(op.value)]]);
            }
            else if (node instanceof nodes_1.ArrNode) {
                const value = builder.json(op.value);
                if (key === '-') {
                    const length = node.length();
                    const after = node.find(length - 1) || node.id;
                    builder.insArr(node.id, after, [value]);
                }
                else {
                    const index = ~~key;
                    if ('' + index !== key)
                        throw new Error('INVALID_INDEX');
                    if (!index)
                        builder.insArr(node.id, node.id, [value]);
                    else {
                        const after = node.find(index - 1);
                        if (!after)
                            throw new Error('NOT_FOUND');
                        builder.insArr(node.id, after, [value]);
                    }
                }
            }
            else
                throw new Error('NOT_FOUND');
        }
    }
    applyRemove(op) {
        const builder = this.builder();
        const steps = (0, util_1.toPath)(op.path);
        if (!steps.length)
            this.setRoot(null);
        else {
            const objSteps = steps.slice(0, steps.length - 1);
            const node = this.model.api.find(objSteps);
            const key = steps[steps.length - 1];
            if (node instanceof nodes_1.ObjNode) {
                const stringKey = String(key);
                if (node.get(stringKey) === undefined)
                    throw new Error('NOT_FOUND');
                builder.insObj(node.id, [[stringKey, builder.const(undefined)]]);
            }
            else if (node instanceof nodes_1.ArrNode) {
                const key = steps[steps.length - 1];
                const index = ~~key;
                if ('' + index !== key)
                    throw new Error('INVALID_INDEX');
                const id = node.find(index);
                if (!id)
                    throw new Error('NOT_FOUND');
                builder.del(node.id, [(0, clock_1.interval)(id, 0, 1)]);
            }
            else
                throw new Error('NOT_FOUND');
        }
    }
    applyReplace(op) {
        const { path, value } = op;
        this.applyRemove({ op: 'remove', path });
        this.applyOpAdd({ op: 'add', path, value });
    }
    applyMove(op) {
        const path = (0, util_1.toPath)(op.path);
        const from = (0, util_1.toPath)(op.from);
        if ((0, json_pointer_1.isChild)(from, path))
            throw new Error('INVALID_CHILD');
        const json = this.json(from);
        this.applyRemove({ op: 'remove', path: from });
        this.applyOpAdd({ op: 'add', path, value: json });
    }
    applyCopy(op) {
        const path = (0, util_1.toPath)(op.path);
        const from = (0, util_1.toPath)(op.from);
        const json = this.json(from);
        this.applyOpAdd({ op: 'add', path, value: json });
    }
    applyTest(op) {
        const path = (0, util_1.toPath)(op.path);
        const json = this.json(path);
        if (!(0, deepEqual_1.deepEqual)(json, op.value))
            throw new Error('TEST');
    }
    applyStrIns(op) {
        const path = (0, util_1.toPath)(op.path);
        const { node } = this.model.api.str(path);
        const length = node.length();
        const after = op.pos ? node.find(length < op.pos ? length - 1 : op.pos - 1) : node.id;
        if (!after)
            throw new Error('OUT_OF_BOUNDS');
        this.builder().insStr(node.id, after, op.str);
    }
    applyStrDel(op) {
        var _a;
        const path = (0, util_1.toPath)(op.path);
        const { node } = this.model.api.str(path);
        const length = node.length();
        if (length <= op.pos)
            return;
        const deletionLength = Math.min((_a = op.len) !== null && _a !== void 0 ? _a : op.str.length, length - op.pos);
        const range = node.findInterval(op.pos, deletionLength);
        if (!range)
            throw new Error('OUT_OF_BOUNDS');
        this.builder().del(node.id, range);
    }
    get(steps) {
        var _a;
        const model = this.model;
        if (!steps.length)
            return model.view();
        else {
            const objSteps = steps.slice(0, steps.length - 1);
            const node = model.api.find(objSteps);
            const key = steps[steps.length - 1];
            if (node instanceof nodes_1.ObjNode) {
                return (_a = node.get(String(key))) === null || _a === void 0 ? void 0 : _a.view();
            }
            else if (node instanceof nodes_1.ArrNode) {
                const index = ~~key;
                if ('' + index !== key)
                    throw new Error('INVALID_INDEX');
                const arrNode = node.getNode(index);
                if (!arrNode)
                    throw new Error('NOT_FOUND');
                return arrNode.view();
            }
        }
        return undefined;
    }
    json(steps) {
        const json = this.get(steps);
        if (json === undefined)
            throw new Error('NOT_FOUND');
        return json;
    }
    setRoot(json) {
        const builder = this.builder();
        builder.root(builder.json(json));
    }
}
exports.JsonPatch = JsonPatch;
