"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AbstractRga = void 0;
const clock_1 = require("../../../json-crdt-patch/clock");
const isUint8Array_1 = require("../../../util/buffers/isUint8Array");
const util_1 = require("../../../util/trees/splay/util");
const util2_1 = require("../../../util/trees/splay/util2");
const util2_2 = require("../../../util/trees/util2");
const printBinary_1 = require("../../../util/print/printBinary");
const printTree_1 = require("../../../util/print/printTree");
const constants_1 = require("../../../json-crdt-patch/constants");
const compareById = (c1, c2) => {
    const ts1 = c1.id;
    const ts2 = c2.id;
    return ts1.sid - ts2.sid || ts1.time - ts2.time;
};
const updateLenOne = (chunk) => {
    const l = chunk.l;
    const r = chunk.r;
    chunk.len = (chunk.del ? 0 : chunk.span) + (l ? l.len : 0) + (r ? r.len : 0);
};
const updateLenOneLive = (chunk) => {
    const l = chunk.l;
    const r = chunk.r;
    chunk.len = chunk.span + (l ? l.len : 0) + (r ? r.len : 0);
};
const dLen = (chunk, delta) => {
    while (chunk) {
        chunk.len += delta;
        chunk = chunk.p;
    }
};
const next = (curr) => {
    const r = curr.r;
    if (r) {
        curr = r;
        let tmp;
        while ((tmp = curr.l))
            curr = tmp;
        return curr;
    }
    let p = curr.p;
    while (p && p.r === curr) {
        curr = p;
        p = p.p;
    }
    return p;
};
const prev = (curr) => {
    const l = curr.l;
    if (l) {
        curr = l;
        let tmp;
        while ((tmp = curr.r))
            curr = tmp;
        return curr;
    }
    let p = curr.p;
    while (p && p.l === curr) {
        curr = p;
        p = p.p;
    }
    return p;
};
class AbstractRga {
    constructor(id) {
        this.id = id;
        this.root = undefined;
        this.ids = undefined;
        this.count = 0;
    }
    ins(after, id, content) {
        const rootId = this.id;
        const afterTime = after.time;
        const afterSid = after.sid;
        const isRootInsert = rootId.time === afterTime && rootId.sid === afterSid;
        if (isRootInsert) {
            this.insAfterRoot(after, id, content);
            return;
        }
        let curr = this.ids;
        let chunk = curr;
        while (curr) {
            const currId = curr.id;
            const currIdSid = currId.sid;
            if (currIdSid > afterSid) {
                curr = curr.l2;
            }
            else if (currIdSid < afterSid) {
                chunk = curr;
                curr = curr.r2;
            }
            else {
                const currIdTime = currId.time;
                if (currIdTime > afterTime) {
                    curr = curr.l2;
                }
                else if (currIdTime < afterTime) {
                    chunk = curr;
                    curr = curr.r2;
                }
                else {
                    chunk = curr;
                    break;
                }
            }
        }
        if (!chunk)
            return;
        const atId = chunk.id;
        const atIdTime = atId.time;
        const atIdSid = atId.sid;
        const atSpan = chunk.span;
        if (atIdSid !== afterSid)
            return;
        const offset = afterTime - atIdTime;
        if (offset >= atSpan)
            return;
        const offsetInInsertAtChunk = afterTime - atIdTime;
        this.insAfterChunk(after, chunk, offsetInInsertAtChunk, id, content);
    }
    insAt(position, id, content) {
        if (!position) {
            const rootId = this.id;
            this.insAfterRoot(rootId, id, content);
            return rootId;
        }
        const found = this.findChunk(position - 1);
        if (!found)
            return undefined;
        const [at, offset] = found;
        const atId = at.id;
        const after = offset === 0 ? atId : new clock_1.Timestamp(atId.sid, atId.time + offset);
        this.insAfterChunk(after, at, offset, id, content);
        return after;
    }
    insAfterRoot(after, id, content) {
        const chunk = this.createChunk(id, content);
        const first = this.first();
        if (!first)
            this.setRoot(chunk);
        else if ((0, clock_1.compare)(first.id, id) < 0)
            this.insertBefore(chunk, first);
        else {
            if ((0, clock_1.containsId)(first.id, first.span, id))
                return;
            this.insertAfterRef(chunk, after, first);
        }
    }
    insAfterChunk(after, chunk, chunkOffset, id, content) {
        const atId = chunk.id;
        const atIdTime = atId.time;
        const atIdSid = atId.sid;
        const atSpan = chunk.span;
        const newChunk = this.createChunk(id, content);
        const needsSplit = chunkOffset + 1 < atSpan;
        if (needsSplit) {
            const idSid = id.sid;
            const idTime = id.time;
            if (atIdSid === idSid && atIdTime <= idTime && atIdTime + atSpan - 1 >= idTime)
                return;
            if (idTime > after.time + 1 || idSid > after.sid) {
                this.insertInside(newChunk, chunk, chunkOffset + 1);
                this.splay(newChunk);
                return;
            }
        }
        this.insertAfterRef(newChunk, after, chunk);
        this.splay(newChunk);
    }
    delete(spans) {
        const length = spans.length;
        for (let i = 0; i < length; i++)
            this.deleteSpan(spans[i]);
        this.onChange();
    }
    deleteSpan(span) {
        const len = span.span;
        const t1 = span.time;
        const t2 = t1 + len - 1;
        const start = this.findById(span);
        if (!start)
            return;
        let chunk = start;
        let last = chunk;
        while (chunk) {
            last = chunk;
            const id = chunk.id;
            const chunkSpan = chunk.span;
            const c1 = id.time;
            const c2 = c1 + chunkSpan - 1;
            if (chunk.del) {
                if (c2 >= t2)
                    break;
                chunk = chunk.s;
                continue;
            }
            const deleteStartsFromLeft = t1 <= c1;
            const deleteStartsInTheMiddle = t1 <= c2;
            if (deleteStartsFromLeft) {
                const deleteFullyContainsChunk = t2 >= c2;
                if (deleteFullyContainsChunk) {
                    chunk.delete();
                    dLen(chunk, -chunk.span);
                    if (t2 <= c2)
                        break;
                }
                else {
                    const range = t2 - c1 + 1;
                    const newChunk = this.split(chunk, range);
                    chunk.delete();
                    updateLenOne(newChunk);
                    dLen(chunk, -chunk.span);
                    break;
                }
            }
            else if (deleteStartsInTheMiddle) {
                const deleteContainsRightSide = t2 >= c2;
                if (deleteContainsRightSide) {
                    const offset = t1 - c1;
                    const newChunk = this.split(chunk, offset);
                    newChunk.delete();
                    newChunk.len = newChunk.r ? newChunk.r.len : 0;
                    dLen(chunk, -newChunk.span);
                    if (t2 <= c2)
                        break;
                }
                else {
                    const right = this.split(chunk, t2 - c1 + 1);
                    const mid = this.split(chunk, t1 - c1);
                    mid.delete();
                    updateLenOne(right);
                    updateLenOne(mid);
                    dLen(chunk, -mid.span);
                    break;
                }
            }
            chunk = chunk.s;
        }
        if (last)
            this.mergeTombstones2(start, last);
    }
    find(position) {
        let curr = this.root;
        while (curr) {
            const l = curr.l;
            const leftLength = l ? l.len : 0;
            let span;
            if (position < leftLength)
                curr = l;
            else if (curr.del) {
                position -= leftLength;
                curr = curr.r;
            }
            else if (position < leftLength + (span = curr.span)) {
                const ticks = position - leftLength;
                const id = curr.id;
                return !ticks ? id : new clock_1.Timestamp(id.sid, id.time + ticks);
            }
            else {
                position -= leftLength + span;
                curr = curr.r;
            }
        }
    }
    findChunk(position) {
        let curr = this.root;
        while (curr) {
            const l = curr.l;
            const leftLength = l ? l.len : 0;
            let span;
            if (position < leftLength)
                curr = l;
            else if (curr.del) {
                position -= leftLength;
                curr = curr.r;
            }
            else if (position < leftLength + (span = curr.span)) {
                return [curr, position - leftLength];
            }
            else {
                position -= leftLength + span;
                curr = curr.r;
            }
        }
    }
    findInterval(position, length) {
        const ranges = [];
        if (!length)
            return ranges;
        let curr = this.root;
        let offset = 0;
        while (curr) {
            const leftLength = curr.l ? curr.l.len : 0;
            if (position < leftLength)
                curr = curr.l;
            else if (curr.del) {
                position -= leftLength;
                curr = curr.r;
            }
            else if (position < leftLength + curr.span) {
                offset = position - leftLength;
                break;
            }
            else {
                position -= leftLength + curr.span;
                curr = curr.r;
            }
        }
        if (!curr)
            return ranges;
        if (curr.span >= length + offset) {
            const id = curr.id;
            ranges.push((0, clock_1.tss)(id.sid, id.time + offset, length));
            return ranges;
        }
        const len = curr.span - offset;
        const id = curr.id;
        ranges.push((0, clock_1.tss)(id.sid, id.time + offset, len));
        length -= len;
        curr = next(curr);
        if (!curr)
            return ranges;
        do {
            if (curr.del)
                continue;
            const id = curr.id;
            const span = curr.span;
            if (length <= span) {
                ranges.push((0, clock_1.tss)(id.sid, id.time, length));
                return ranges;
            }
            ranges.push((0, clock_1.tss)(id.sid, id.time, span));
            length -= span;
        } while ((curr = next(curr)) && length > 0);
        return ranges;
    }
    findInterval2(from, to) {
        const ranges = [];
        this.range0(undefined, from, to, (chunk, off, len) => {
            const id = chunk.id;
            ranges.push((0, clock_1.tss)(id.sid, id.time + off, len));
        });
        return ranges;
    }
    range0(startChunk, from, to, callback) {
        let chunk = startChunk ? startChunk : this.findById(from);
        if (startChunk)
            while (chunk && !(0, clock_1.containsId)(chunk.id, chunk.span, from))
                chunk = next(chunk);
        if (!chunk)
            return;
        if (!chunk.del) {
            const off = from.time - chunk.id.time;
            const toContainedInChunk = (0, clock_1.containsId)(chunk.id, chunk.span, to);
            if (toContainedInChunk) {
                const len = to.time - from.time + 1;
                callback(chunk, off, len);
                return chunk;
            }
            const len = chunk.span - off;
            callback(chunk, off, len);
        }
        else {
            if ((0, clock_1.containsId)(chunk.id, chunk.span, to))
                return;
        }
        chunk = next(chunk);
        while (chunk) {
            const toContainedInChunk = (0, clock_1.containsId)(chunk.id, chunk.span, to);
            if (toContainedInChunk) {
                if (!chunk.del)
                    callback(chunk, 0, to.time - chunk.id.time + 1);
                return chunk;
            }
            if (!chunk.del)
                callback(chunk, 0, chunk.span);
            chunk = next(chunk);
        }
        return chunk;
    }
    first() {
        let curr = this.root;
        while (curr) {
            const l = curr.l;
            if (l)
                curr = l;
            else
                return curr;
        }
        return curr;
    }
    last() {
        let curr = this.root;
        while (curr) {
            const r = curr.r;
            if (r)
                curr = r;
            else
                return curr;
        }
        return curr;
    }
    lastId() {
        const chunk = this.last();
        if (!chunk)
            return undefined;
        const id = chunk.id;
        const span = chunk.span;
        return span === 1 ? id : new clock_1.Timestamp(id.sid, id.time + span - 1);
    }
    next(curr) {
        return next(curr);
    }
    length() {
        const root = this.root;
        return root ? root.len : 0;
    }
    size() {
        return this.count;
    }
    pos(chunk) {
        const p = chunk.p;
        const l = chunk.l;
        if (!p)
            return l ? l.len : 0;
        const parentPos = this.pos(p);
        const isRightChild = p.r === chunk;
        if (isRightChild)
            return parentPos + (p.del ? 0 : p.span) + (l ? l.len : 0);
        const r = chunk.r;
        return parentPos - (chunk.del ? 0 : chunk.span) - (r ? r.len : 0);
    }
    setRoot(chunk) {
        this.root = chunk;
        this.insertId(chunk);
        this.onChange();
    }
    insertBefore(chunk, before) {
        const l = before.l;
        before.l = chunk;
        chunk.l = l;
        chunk.p = before;
        let lLen = 0;
        if (l) {
            l.p = chunk;
            lLen = l.len;
        }
        chunk.len = chunk.span + lLen;
        dLen(before, chunk.span);
        this.insertId(chunk);
        this.onChange();
    }
    insertAfter(chunk, after) {
        const r = after.r;
        after.r = chunk;
        chunk.r = r;
        chunk.p = after;
        let rLen = 0;
        if (r) {
            r.p = chunk;
            rLen = r.len;
        }
        chunk.len = chunk.span + rLen;
        dLen(after, chunk.span);
        this.insertId(chunk);
        this.onChange();
    }
    insertAfterRef(chunk, ref, left) {
        const id = chunk.id;
        const sid = id.sid;
        const time = id.time;
        let isSplit = false;
        for (;;) {
            const leftId = left.id;
            const leftNextTick = leftId.time + left.span;
            if (!left.s) {
                isSplit = leftId.sid === sid && leftNextTick === time && leftNextTick - 1 === ref.time;
                if (isSplit)
                    left.s = chunk;
            }
            const right = next(left);
            if (!right)
                break;
            const rightId = right.id;
            const rightIdTime = rightId.time;
            const rightIdSid = rightId.sid;
            if (rightIdTime < time)
                break;
            if (rightIdTime === time) {
                if (rightIdSid === sid)
                    return;
                if (rightIdSid < sid)
                    break;
            }
            left = right;
        }
        if (isSplit && !left.del) {
            this.mergeContent(left, chunk.data);
            left.s = undefined;
        }
        else
            this.insertAfter(chunk, left);
    }
    mergeContent(chunk, content) {
        const span1 = chunk.span;
        chunk.merge(content);
        dLen(chunk, chunk.span - span1);
        this.onChange();
        return;
    }
    insertInside(chunk, at, offset) {
        const p = at.p;
        const l = at.l;
        const r = at.r;
        const s = at.s;
        const len = at.len;
        const at2 = at.split(offset);
        at.s = at2;
        at2.s = s;
        at.l = at.r = at2.l = at2.r = undefined;
        at2.l = undefined;
        chunk.p = p;
        if (!l) {
            chunk.l = at;
            at.p = chunk;
        }
        else {
            chunk.l = l;
            l.p = chunk;
            const a = l.r;
            l.r = at;
            at.p = l;
            at.l = a;
            if (a)
                a.p = at;
        }
        if (!r) {
            chunk.r = at2;
            at2.p = chunk;
        }
        else {
            chunk.r = r;
            r.p = chunk;
            const b = r.l;
            r.l = at2;
            at2.p = r;
            at2.r = b;
            if (b)
                b.p = at2;
        }
        if (!p)
            this.root = chunk;
        else if (p.l === at)
            p.l = chunk;
        else
            p.r = chunk;
        updateLenOne(at);
        updateLenOne(at2);
        if (l)
            l.len = (l.l ? l.l.len : 0) + at.len + (l.del ? 0 : l.span);
        if (r)
            r.len = (r.r ? r.r.len : 0) + at2.len + (r.del ? 0 : r.span);
        chunk.len = len + chunk.span;
        const span = chunk.span;
        let curr = chunk.p;
        while (curr) {
            curr.len += span;
            curr = curr.p;
        }
        this.insertId(at2);
        this.insertIdFast(chunk);
        this.onChange();
    }
    split(chunk, ticks) {
        const s = chunk.s;
        const newChunk = chunk.split(ticks);
        const r = chunk.r;
        chunk.s = newChunk;
        newChunk.r = r;
        newChunk.s = s;
        chunk.r = newChunk;
        newChunk.p = chunk;
        this.insertId(newChunk);
        if (r)
            r.p = newChunk;
        return newChunk;
    }
    mergeTombstones(ch1, ch2) {
        if (!ch1.del || !ch2.del)
            return false;
        const id1 = ch1.id;
        const id2 = ch2.id;
        if (id1.sid !== id2.sid)
            return false;
        if (id1.time + ch1.span !== id2.time)
            return false;
        ch1.s = ch2.s;
        ch1.span += ch2.span;
        this.deleteChunk(ch2);
        return true;
    }
    mergeTombstones2(start, end) {
        let curr = start;
        while (curr) {
            const nextCurr = next(curr);
            if (!nextCurr)
                break;
            const merged = this.mergeTombstones(curr, nextCurr);
            if (!merged) {
                if (nextCurr === end) {
                    if (nextCurr) {
                        const n = next(nextCurr);
                        if (n)
                            this.mergeTombstones(nextCurr, n);
                    }
                    break;
                }
                curr = curr.s;
            }
        }
        const left = prev(start);
        if (left)
            this.mergeTombstones(left, start);
    }
    removeTombstones() {
        let curr = this.first();
        const list = [];
        while (curr) {
            if (curr.del)
                list.push(curr);
            curr = next(curr);
        }
        for (let i = 0; i < list.length; i++)
            this.deleteChunk(list[i]);
    }
    deleteChunk(chunk) {
        this.deleteId(chunk);
        const p = chunk.p;
        const l = chunk.l;
        const r = chunk.r;
        chunk.id = constants_1.ORIGIN;
        if (!l && !r) {
            if (!p)
                this.root = undefined;
            else {
                if (p.l === chunk)
                    p.l = undefined;
                else
                    p.r = undefined;
            }
        }
        else if (l && r) {
            let mostRightChildFromLeft = l;
            while (mostRightChildFromLeft.r)
                mostRightChildFromLeft = mostRightChildFromLeft.r;
            mostRightChildFromLeft.r = r;
            r.p = mostRightChildFromLeft;
            const rLen = r.len;
            let curr;
            curr = mostRightChildFromLeft;
            if (!p) {
                this.root = l;
                l.p = undefined;
            }
            else {
                if (p.l === chunk)
                    p.l = l;
                else
                    p.r = l;
                l.p = p;
            }
            while (curr && curr !== p) {
                curr.len += rLen;
                curr = curr.p;
            }
        }
        else {
            const child = (l || r);
            child.p = p;
            if (!p)
                this.root = child;
            else if (p.l === chunk)
                p.l = child;
            else
                p.r = child;
        }
    }
    insertId(chunk) {
        this.ids = (0, util2_2.insert2)(this.ids, chunk, compareById);
        this.count++;
        this.ids = (0, util2_1.splay2)(this.ids, chunk);
    }
    insertIdFast(chunk) {
        this.ids = (0, util2_2.insert2)(this.ids, chunk, compareById);
        this.count++;
    }
    deleteId(chunk) {
        this.ids = (0, util2_2.remove2)(this.ids, chunk);
        this.count--;
    }
    findById(after) {
        const afterSid = after.sid;
        const afterTime = after.time;
        let curr = this.ids;
        let chunk = curr;
        while (curr) {
            const currId = curr.id;
            const currIdSid = currId.sid;
            if (currIdSid > afterSid) {
                curr = curr.l2;
            }
            else if (currIdSid < afterSid) {
                chunk = curr;
                curr = curr.r2;
            }
            else {
                const currIdTime = currId.time;
                if (currIdTime > afterTime) {
                    curr = curr.l2;
                }
                else if (currIdTime < afterTime) {
                    chunk = curr;
                    curr = curr.r2;
                }
                else {
                    chunk = curr;
                    break;
                }
            }
        }
        if (!chunk)
            return;
        const atId = chunk.id;
        const atIdTime = atId.time;
        const atIdSid = atId.sid;
        const atSpan = chunk.span;
        if (atIdSid !== afterSid)
            return;
        if (afterTime < atIdTime)
            return;
        const offset = afterTime - atIdTime;
        if (offset >= atSpan)
            return;
        return chunk;
    }
    splay(chunk) {
        const p = chunk.p;
        if (!p)
            return;
        const pp = p.p;
        const l2 = p.l === chunk;
        if (!pp) {
            if (l2)
                (0, util_1.rSplay)(chunk, p);
            else
                (0, util_1.lSplay)(chunk, p);
            this.root = chunk;
            updateLenOne(p);
            updateLenOneLive(chunk);
            return;
        }
        const l1 = pp.l === p;
        if (l1) {
            if (l2) {
                this.root = (0, util_1.llSplay)(this.root, chunk, p, pp);
            }
            else {
                this.root = (0, util_1.lrSplay)(this.root, chunk, p, pp);
            }
        }
        else {
            if (l2) {
                this.root = (0, util_1.rlSplay)(this.root, chunk, p, pp);
            }
            else {
                this.root = (0, util_1.rrSplay)(this.root, chunk, p, pp);
            }
        }
        updateLenOne(pp);
        updateLenOne(p);
        updateLenOneLive(chunk);
        this.splay(chunk);
    }
    iterator() {
        let curr = this.first();
        return () => {
            const res = curr;
            if (curr)
                curr = next(curr);
            return res;
        };
    }
    ingest(size, next) {
        if (size < 1)
            return;
        const splitLeftChunks = new Map();
        this.root = this._ingest(size, () => {
            const chunk = next();
            const id = chunk.id;
            const key = id.sid + '.' + id.time;
            const split = splitLeftChunks.get(key);
            if (split) {
                split.s = chunk;
                splitLeftChunks.delete(key);
            }
            const nextStampAfterSpan = (0, clock_1.tick)(id, chunk.span);
            splitLeftChunks.set(nextStampAfterSpan.sid + '.' + nextStampAfterSpan.time, chunk);
            return chunk;
        });
    }
    _ingest(size, next) {
        const leftSize = size >> 1;
        const rightSize = size - leftSize - 1;
        const c1 = leftSize > 0 ? this._ingest(leftSize, next) : undefined;
        const c2 = next();
        if (c1) {
            c2.l = c1;
            c1.p = c2;
        }
        const c3 = rightSize > 0 ? this._ingest(rightSize, next) : undefined;
        if (c3) {
            c2.r = c3;
            c3.p = c2;
        }
        updateLenOne(c2);
        this.insertId(c2);
        return c2;
    }
    toStringName() {
        return this.constructor.name;
    }
    toString(tab = '') {
        const view = this.view();
        let value = '';
        if ((0, isUint8Array_1.isUint8Array)(view))
            value += ` { ${('' + view).replaceAll(',', ', ')} }`;
        else if (typeof view === 'string')
            value += `{ ${view.length > 32 ? JSON.stringify(view.substring(0, 32)) + ' …' : JSON.stringify(view)} }`;
        const header = `${this.toStringName()} ${(0, clock_1.toDisplayString)(this.id)} ${value}`;
        return header + (0, printTree_1.printTree)(tab, [(tab) => (this.root ? this.printChunk(tab, this.root) : '∅')]);
    }
    printChunk(tab, chunk) {
        return (this.formatChunk(chunk) +
            (0, printBinary_1.printBinary)(tab, [
                chunk.l ? (tab) => this.printChunk(tab, chunk.l) : null,
                chunk.r ? (tab) => this.printChunk(tab, chunk.r) : null,
            ]));
    }
    formatChunk(chunk) {
        const id = (0, clock_1.toDisplayString)(chunk.id);
        let str = `${chunk.constructor.name} ${id}!${chunk.span} len:${chunk.len}`;
        if (chunk.del)
            str += ` [${chunk.span}]`;
        else {
            if ((0, isUint8Array_1.isUint8Array)(chunk.data))
                str += ` { ${chunk.data.toString().replaceAll(',', ', ')} }`;
            else if (typeof chunk.data === 'string') {
                const data = chunk.data.length > 32 ? JSON.stringify(chunk.data.substring(0, 32)) + ' …' : JSON.stringify(chunk.data);
                str += ` { ${data} }`;
            }
        }
        return str;
    }
}
exports.AbstractRga = AbstractRga;
