"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpSplit = void 0;
const AbstractOp_1 = require("./AbstractOp");
const json_pointer_1 = require("../../json-pointer");
const util_1 = require("../util");
class OpSplit extends AbstractOp_1.AbstractOp {
    constructor(path, pos, props) {
        super(path);
        this.pos = pos;
        this.props = props;
    }
    op() {
        return 'split';
    }
    code() {
        return 10;
    }
    apply(doc) {
        const ref = (0, json_pointer_1.find)(doc, this.path);
        if (ref.val === undefined)
            throw new Error('NOT_FOUND');
        const tuple = this.split(ref.val);
        if ((0, json_pointer_1.isObjectReference)(ref))
            ref.obj[ref.key] = tuple;
        else if ((0, json_pointer_1.isArrayReference)(ref)) {
            ref.obj[ref.key] = tuple[0];
            ref.obj.splice(ref.key + 1, 0, tuple[1]);
        }
        else
            doc = tuple;
        return { doc, old: ref.val };
    }
    split(node) {
        if (typeof node === 'string') {
            const { pos, props } = this;
            const before = node.slice(0, pos);
            const after = node.slice(pos);
            if (!props)
                return [before, after];
            const textNodes = [
                Object.assign(Object.assign({}, props), { text: before }),
                Object.assign(Object.assign({}, props), { text: after }),
            ];
            return textNodes;
        }
        else if ((0, util_1.isTextNode)(node)) {
            const { pos, props } = this;
            const before = node.text.slice(0, pos);
            const after = node.text.slice(pos);
            const textNodes = [
                Object.assign(Object.assign(Object.assign({}, node), props), { text: before }),
                Object.assign(Object.assign(Object.assign({}, node), props), { text: after }),
            ];
            return textNodes;
        }
        else if ((0, util_1.isElementNode)(node)) {
            const { pos, props } = this;
            const before = node.children.slice(0, pos);
            const after = node.children.slice(pos);
            const elementNodes = [
                Object.assign(Object.assign(Object.assign({}, node), props), { children: before }),
                Object.assign(Object.assign(Object.assign({}, node), props), { children: after }),
            ];
            return elementNodes;
        }
        else if (typeof node === 'number') {
            const { pos } = this;
            return [pos, node - pos];
        }
        else
            return [node, node];
    }
    toJson(parent) {
        const op = {
            op: 'split',
            path: (0, json_pointer_1.formatJsonPointer)(this.path),
            pos: this.pos,
        };
        if (this.props)
            op.props = this.props;
        return op;
    }
    toCompact(parent, verbose) {
        const opcode = verbose ? 'split' : 10;
        return this.props ? [opcode, this.path, this.pos, this.props] : [opcode, this.path, this.pos];
    }
    encode(encoder, parent) {
        encoder.encodeArrayHeader(this.props ? 4 : 3);
        encoder.writer.u8(10);
        encoder.encodeArray(this.path);
        encoder.encodeNumber(this.pos);
        if (this.props)
            encoder.encodeObject(this.props);
    }
}
exports.OpSplit = OpSplit;
