"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.toText = void 0;
const wordWrap_1 = require("../../util/strings/wordWrap");
const util_1 = require("./util");
const formatComment = (comment, __) => {
    if (!comment)
        return '';
    const lines = (0, wordWrap_1.wordWrap)(comment, { width: 80 - 3 - __.length });
    return __ + '/**\n' + __ + ' * ' + lines.join('\n' + __ + ' * ') + '\n' + __ + ' */\n';
};
const toText = (node, __ = '') => {
    if (Array.isArray(node))
        return node.map((s) => (0, exports.toText)(s, __)).join('\n');
    const ____ = __ + util_1.TAB;
    switch (node.node) {
        case 'ModuleDeclaration': {
            let out = '';
            out += `${__}namespace ${node.name} {\n`;
            out += (0, exports.toText)(node.statements, ____);
            out += `${__}}\n`;
            return out;
        }
        case 'InterfaceDeclaration': {
            const { name, members, comment } = node;
            let out = '';
            out += formatComment(comment, __);
            out += `${__}interface ${name} {\n`;
            out += (0, exports.toText)(members, ____);
            out += `\n${__}}\n`;
            return out;
        }
        case 'PropertySignature': {
            const name = (0, util_1.normalizeKey)(node.name);
            let out = '';
            out += formatComment(node.comment, __);
            return out + `${__}${name}${node.optional ? '?' : ''}: ${(0, exports.toText)(node.type, __)};`;
        }
        case 'IndexSignature': {
            return `${__}[key: string]: ${(0, exports.toText)(node.type, __)};`;
        }
        case 'TypeAliasDeclaration': {
            let out = '';
            out += formatComment(node.comment, __);
            out += `${__}type ${node.name} = ${(0, exports.toText)(node.type)};\n`;
            return out;
        }
        case 'ArrayType': {
            const simple = (0, util_1.isSimpleType)(node.elementType);
            const inner = (0, exports.toText)(node.elementType, __);
            return simple ? `${inner}[]` : `Array<${inner}>`;
        }
        case 'TupleType': {
            const hasObject = node.elements.some((e) => e.node === 'TypeLiteral');
            if (hasObject) {
                return `[\n${____}${node.elements.map((e) => (0, exports.toText)(e, ____)).join(',\n' + ____)}\n${__}]`;
            }
            else
                return `[${node.elements.map((e) => (0, exports.toText)(e, __)).join(', ')}]`;
        }
        case 'GenericTypeAnnotation': {
            return node.id.name;
        }
        case 'StringKeyword': {
            return 'string';
        }
        case 'NumberKeyword': {
            return 'number';
        }
        case 'BooleanKeyword': {
            return 'boolean';
        }
        case 'NullKeyword': {
            return 'null';
        }
        case 'AnyKeyword': {
            return 'any';
        }
        case 'UnknownKeyword': {
            return 'unknown';
        }
        case 'TypeLiteral': {
            return `{\n${(0, exports.toText)(node.members, ____)}\n${__}}`;
        }
        case 'StringLiteral': {
            return JSON.stringify(node.text);
        }
        case 'NumericLiteral': {
            return node.text;
        }
        case 'TrueKeyword': {
            return 'true';
        }
        case 'FalseKeyword': {
            return 'false';
        }
        case 'UnionType': {
            return node.types.map((t) => (0, exports.toText)(t, ____)).join(' | ');
        }
        case 'TypeReference': {
            return node.typeName;
        }
    }
    return '';
};
exports.toText = toText;
